/*******************************************************************************

KSData.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2011 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

#ifndef KSData_h
#define KSData_h

#ifdef WINRT_ENV
#include <string>
#endif

	// Lua
#include <AgKernel/lua.h>

	 // wichitafoundation/core
#include "WFApiDefs.h"

	// wichitafoundation/core/threading
#include "WFMessagingData.h"

//==============================================================================
/*
	KSData is an abstract container around an immutable sequence of bytes.
	It is designed to be efficiently transferred among Lua universes and between
	Lua and native code.

	The implementation of KSData varies from platform to platform. This interface
	is intended to represent a fairly minimal contract that should be easily
	implemented on any likely target platform.

	Some of the details of transiting data among universes is intentionally
	punted off to platform-specific code.

	KSData also exports a namespace that is available to Lua:

		<pre>local KSData = require 'KSData'  -- or require 'core.lua.KSData'?

		local data = KSData.make( source )
			-- source can be either an existing data or a string

		local isData = KSData.isData( data )
			-- Returns true if this is a KSData object.
			-- Returns false if any other value type.

		local str = tostring( data )

		local len = KSData.length( data )
			-- returns #bytes; returns zero if this isn't a data object	</pre>

*/

//------------------------------------------------------------------------------

/*!
	Makes a copy of the supplied data as a KSData object and
	pushes a reference to that object onto the Lua stack.

	Throws an exception on error.

	\param L The Lua state.
	\param data The data.
	\param size The size of the data in bytes.
*/

WF_API
void KSData_pushReferenceToData( lua_State* L, const void* data, size_t size );

//------------------------------------------------------------------------------

/*!
 	Creates a mutable data object on the Lua stack and returns a pointer to
	the data contained in the object, which can then be filled in by the caller.

	This can be risky, but can avoid excess copying. Be careful that you don't
	continue to treat the data as mutable after handing it off.

	Returns NULL on error. Throws an exception if unable to allocate the
	Lua stub.

*/

WF_API
void* KSData_pushMutableData( lua_State* L, size_t size );

//------------------------------------------------------------------------------

/*!
	Pushes a data object that represents the read-only mapping
	of a file referenced in a Lua state onto the Lua stack.

	\param L The Lua state.
	\param pathIndex Index of the file path in the Lua state.

	\return  1 on success, 0 on failure.
*/

WF_API
int KSData_pushMemoryMappedContentsOfFileWithPathAtIndex(
				lua_State* L, int pathIndex );

//------------------------------------------------------------------------------

/*!
	Pushes a data object referencing the given messaging data object.

	\param L The Lua state.
	\param dataRef The message data to reference.

	\return  1 on success, 0 on failure.
*/

WF_API
int KSData_pushReferenceToMessagingData( lua_State* L, WFMessaging_DataRef dataRef );

//------------------------------------------------------------------------------

/*!
	If the object at the stack index is a KSData, then extract it's contents as
	a WFMessaging_DataRef. This does not retain the data object. Returns NULL
	if unsuccessful.

	\param L The Lua state.
	\param index The index of the KSData object.

	\return  The messaging data if the map goes through successfully. NULL otherwise.
*/

WF_API
WFMessaging_DataRef KSData_toWFMData( lua_State* L, int index );

//------------------------------------------------------------------------------

/*!
	Pushes a clone of a KSData object in a Lua state onto the Lua stack.
	Clones can be closed without invalidating the existing object.

	\param L The Lua state.
	\param index Index of the object in the Lua state.

*/

WF_API
int KSData_pushClone( lua_State* L, int index );

//------------------------------------------------------------------------------

/*!
    	Closes a KSData object at index.

	\param L The Lua state.
	\param index Index of the object in the Lua state. If this is
		not a KSData object, the call does nothing.
*/

WF_API
void KSData_close( lua_State* L, int index );

//------------------------------------------------------------------------------

/*!
	Reports whether an object in a Lua state is a KSData object reference.

	\param L The Lua state.
	\param index Index of the object in the Lua state.

	\return Non-zero if the value at the specified index is a KSData
	 	object reference, 0 if any other data type.
*/

WF_API
int KSData_isData( lua_State* L, int index );

//------------------------------------------------------------------------------

/*!
	Retrieves the length of a KSData object.

	\param L The Lua state.
	\param index Index of the object in the Lua state.

 	\return The length in bytes, or 0 if this is any other data type.

*/

WF_API
size_t KSData_length( lua_State* L, int index );

//------------------------------------------------------------------------------

WF_API
int KSData_isData_extended(
			lua_State* L,
			int index,
			const void** bytes,
			size_t* length );
	// Returns true (non-zero) if the value at the specified index is a KSData
	// object reference. Returns false (zero) if any other data type. If it
	// returns true, it also fills in bytes and length.

//------------------------------------------------------------------------------

/*!
	Reports whether an object in a Lua state is a KSData object reference,
	and if it is, retrieves the contained data.

	\param L The Lua state.
	\param index Index of the object in the Lua state.
	\param bytes A buffer in which to return the data.
	\parem length A buffer in which to return the length of the data in bytes.


	\return Non-zero if the value at the specified index is a KSData
	 object reference. Returns false (zero) if any other data type.
*/

WF_API
size_t KSData_copyTo( lua_State* L, int index, void* buffer, size_t bufSize );

//------------------------------------------------------------------------------

#if IOS_ENV || MAC_ENV

//------------------------------------------------------------------------------

#include <CoreFoundation/CoreFoundation.h>

//------------------------------------------------------------------------------

/*!
	Extracts a CFData from a KSData.

	\param L The Lua state.
	\param index Index of the object in the Lua state.

	\return The extracted data if the object is a KSData object, NULL otherwise.

*/

WF_API
CFDataRef KSData_toCFData( lua_State* L, int index );

//------------------------------------------------------------------------------

/*!
	Pushes a CFData onto the Lua stack.

	\param L The Lua state.
	\param dataRef The data object. If NULL, pushes nil on the stack.
*/

WF_API
void KSData_pushReferenceToCFData( lua_State* L, CFDataRef dataRef );

//------------------------------------------------------------------------------

#if __OBJC__

#import <Foundation/NSData.h>

/*!
	Pushes an NSData onto the Lua stack.  For ARC code, this is nicer than
	KSData_pushReferenceToCFData because although NSData and CSDataRef
	are "toll-free bridged", for lifetime issues NSData is easier to deal
	with (you don't need __bridge).

	\param L The Lua state.
	\param dataRef The data object. If NULL, pushes nil on the stack.
*/

WF_API
void KSData_pushNSData( lua_State* L, NSData* dataRef );

#endif // __OBJC__

//------------------------------------------------------------------------------

#endif // IOS_ENV || MAC_ENV

//------------------------------------------------------------------------------

#endif // KSData_h
