/*******************************************************************************

KSLua.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2010 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

#ifndef KSLua_h
#define KSLua_h

#include <AgKernel/lua.h>

#include "WFApiDefs.h"


//==============================================================================

#pragma mark -
#pragma mark Standard utility routines for Lua

//------------------------------------------------------------------------------
/*
	Standard utility routines for Lua. View this as an extension of lua.h and
	lauxlib.h. Error handling, etc. all parallels those modules.
*/

//------------------------------------------------------------------------------

WF_API int ksl_normalize( lua_State* L, int index );
	// Normalize a stack index so that it is always a positive number.
	// If already positive, return it unchanged; if negative, apply it to
	// existing stack top to return a positive index. Also accounts for
	// magic indices.

//------------------------------------------------------------------------------

WF_API const char* ksl_toString( lua_State* L, int index );
	// Roughly equivalent to calling tostring() in Lua; is guaranteed to always
	// return a string, even if that string is "nil".
	
	// WARNING: This call will change the value of the stack at this index if it
	// is not already a string!

//------------------------------------------------------------------------------

WF_API int ksl_requireAndPcall( lua_State* L, const char* packageName, const char* functionName,
					int nArgs, int nResults );

	// Roughly the equivalent of this (hybrid Lua/C pseudocode):
		// local package = assert( require( packageName ) )
		// local func = assert( package[ functionName ] )
		// return lua_pcall( func( ... ) )

	// Returns zero if the function is found and completes without throwing
	// a Lua error.
	
	// Returns a Lua error code if anything else happens. If so, a description
	// of the error (typically a string) will be left at the top of the stack
	// (i.e. just like lua_pcall()).


//==============================================================================

#pragma mark -
#pragma mark Error handling functions

//------------------------------------------------------------------------------

WF_API void ksl_throwCancelError( lua_State* L );
	// Throw a Lua error that can be identified as such. Does not return.

//------------------------------------------------------------------------------

WF_API int ksl_isCancelError( lua_State* L, int index );
	// Return 1 if the Lua value at the stack was thrown via ksl_throwCancelError.
	// Return 0 if not.


//==============================================================================

#pragma mark -
#pragma mark Cache utilities

//------------------------------------------------------------------------------

WF_API int ksl_fetchCached( lua_State * L, void * cookie );
	// TODO [mhamburg 2013-06-14 via scouten]: API documentation, please?

//------------------------------------------------------------------------------

WF_API void ksl_cacheStackTop( lua_State * L, void * cookie );
	// TODO [mhamburg 2013-06-14 via scouten]: API documentation, please?


//==============================================================================

#pragma mark -
#pragma mark Table construction utilities

//------------------------------------------------------------------------------

WF_API int ksl_pushOrCreateTable( lua_State* L, int sourceIndex, const void* key );
	// Push a table or create one based on looking up the light userdata key in
	// the given source table (often LUA_REGISTRYINDEX). If this creates the table,
	// it returns 1 so that the caller can do further initialization. It returns
	// 0 if the table already existed (in fact if anything with that key
	// exists).

//------------------------------------------------------------------------------

WF_API int ksl_pushOrCreateTableWithIndexMetamethodAndMode(
				lua_State* L, int sourceIndex, const void* key,
				lua_CFunction indexFunc, const char* mode );
	// Push a table or create one based on looking up the light userdata key in
	// the given source table (often LUA_REGISTRYINDEX). If this creates the table,
	// it returns 1 so that the caller can do further initialization. It returns
	// 0 if the table already existed (in fact if anything with that key
	// exists). If it does create a table, it gives a metatable with the
	// supplied __index function and __mode value if mode is not NULL. If neither
	// the index function nor the mode is supplied, this acts just like
	// ksl_pushOrCreateTable.

//------------------------------------------------------------------------------

WF_API void ksl_tableSetFullyWeakMetatable( lua_State* L );
	// Set a fully weak metatable { __mode = 'kv' } on the item on the top of
	// the stack which is expected to be a table. (If the top of the stack
	// isn't a table, this routine logs an error and does nothing.)
	
//------------------------------------------------------------------------------

WF_API int ksl_pushOrCreateFullyWeakTable( lua_State* L, int sourceIndex, const void* key );
	// The combination of the above two routines to handle the common case
	// of needing a fully-weak table. This still returns a flag to indicate
	// whether it created the table.

//------------------------------------------------------------------------------

WF_API void ksl_pushOrConstructTableForInfo(
				lua_State* L,
				int sourceIndex,
				const void* info,
				void (*initTable)( lua_State* L, int tableIndex, const void* info ) );
	// If we have already constructed a table for info as registered in the source
	// table, then just leave it on the stack. Otherwise, attempt to construct one
	// by creating a new table, calling initTable, and if that doesn't throw
	// registering and returning the table.


//==============================================================================

#pragma mark -
#pragma mark Function construction utilities

//------------------------------------------------------------------------------

WF_API void ksl_pushConstantFunction( lua_State* L );
	// Push a function that returns the value currently at the top of the
	// stack. This removes that value and replaces it with the function.
	// The stack must be non-empty.


//==============================================================================

#pragma mark -
#pragma mark Cached function utilities

//------------------------------------------------------------------------------

WF_API void ksl_pushCachedCFunction(
				lua_State * L, void * cookie, lua_CFunction func );
	// Lookup or create a cached C function.

//------------------------------------------------------------------------------

WF_API void ksl_pushCachedLuaFunction(
					lua_State * L,
					void * cookie,
					const char * modname,
					const char * funcname );
	// Lookup and push the function for modname.funcname. Throws an error if
	// the function cannot be found.


//==============================================================================

#pragma mark -
#pragma mark Metatable utilities

//------------------------------------------------------------------------------

WF_API void ksl_indexMetamethodStoreValue( lua_State* L );
	// Assume we're in the midst of an __index metamethod and the top of the
	// stack has the computed value. Store this into the appropriate location
	// in the indexed table and leave the stack as it is.
	
	// Stack state: ( tk ... result )
	// Stores t[ k ] = result. Leaves the stack unchanged.
	
	// Useful if you've computed the result for an index metamethod and then
	// want to store it in the table to avoid recomputation.

//------------------------------------------------------------------------------
/*
	These functions set the __tostring and/or __metatable entries in a
	metatable to the string specified by fmt and any arguments that follow.
	If __metatable is set, this prevents Lua code from accessing the actual
	metatable entry.

	Just for clarity, these functions do not create a closure; they evaluate
	the format string immediately and store the constant string result in the
	designated table locations.
*/

WF_API void ksl_setToStringMetamethodf( lua_State* L, int mtIndex, const char* fmt, ... );
	// Set the __tostring method entry in a metatable.

WF_API void ksl_setToStringMetamethodv( lua_State* L, int mtIndex, const char* fmt, va_list va );
	// Set the __tostring method entry in a metatable.
	
WF_API void ksl_protectMetatablef( lua_State* L, int mtIndex, const char* fmt, ... );
	// Set the __metatable method entry in a metatable.

WF_API void ksl_protectMetatablev( lua_State* L, int mtIndex, const char* fmt, va_list va );
	// Set the __metatable method entry in a metatable.

WF_API void ksl_setToStringMetamethodAndProtectMetatablef( lua_State* L, int mtIndex, const char* fmt, ... );
	// Set the __tostring and __metatable entries in a metatable.
	
WF_API void ksl_setToStringMetamethodAndProtectMetatablev( lua_State* L, int mtIndex, const char* fmt, va_list va );
	// Set the __tostring and __metatable entries in a metatable.


//==============================================================================

#pragma mark -
#pragma mark Debugging utilities

//------------------------------------------------------------------------------

WF_API const char* ksl_toStringForDebug( lua_State* L, int index );
	// Convert the value on the Lua stack at this index to a string (added to
	// top of stack) that is suitable for debug printing. Also return a pointer
	// to this string, which is valid so long as the value remains on the Lua stack.

//------------------------------------------------------------------------------

WF_API const char* ksl_stackToString( lua_State* L );
	// Convert all values on the Lua stack into strings suitable for debugging
	// (see ksl_toStringForDebug). Concat these debug strings together into
	// one big string, one value per line. Also return a pointer to the string,
	// which is valid so long as the value remains on the Lua stack.
	// Adds one value to Lua stack (the string).


//==============================================================================

#pragma mark -
#pragma mark GC hooks

//------------------------------------------------------------------------------
/*
	Lua only knows about sizeof( void* ) for indirect proxies such as are used
	for Objective-C objects. Often this is fine, but if the proxy points to a
	large object like an image, we really need to let the garbage collector know
	about this so that it can go faster (or go slower if we remove pressure).
	The exact behavior of this routine depends on the behavior of the Lua GC,
	but this provides a standardized entrypoint for other code to inform Lua
	about memory usage that it cannot directly observe.
*/

//------------------------------------------------------------------------------

WF_API void ksl_adjustMemoryPressure( lua_State* L, int delta );
	// Account for delta bytes being allocated (positive) or released (negative).
	// It is an undetected but very bad error to report space released that was
	// not previously reported allocated.

//==============================================================================

#endif
