/*******************************************************************************

KSLuaObjCBridge.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2010 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

#ifndef KSLuaObjCBridge_h
#define KSLuaObjCBridge_h

#import <Foundation/NSObject.h>

#import <AgKernel/lua.h>

#import <WFCore/WFApiDefs.h>


//==============================================================================

@interface NSObject( KSLuaObjCProxy )

//------------------------------------------------------------------------------

- (NSObject*) didCreateLuaProxy: (lua_State*) L;
	// Notification that we just created a Lua proxy for the object. The proxy
	// is on the top of the stack. On NSObject, this calls retain and overrides
	// should call through.

//------------------------------------------------------------------------------

- (void) didCloseLuaProxy;
	// Notification that we destroyed a reference to this object from Lua.
	// The Lua state is in the midst of executing a GC metamethod though most
	// uses can just ignore it.
	
	// On NSObject, this calls release and overrides should call through to the
	// superclass.

//------------------------------------------------------------------------------

@end

//==============================================================================

WF_API void kslobjc_pushNSObjectProxy( lua_State* L, NSObject* object );
	// Push the appropriate object proxy for the object.

WF_API BOOL kslobjc_pushNSObjectProxyIfExists( lua_State* L, NSObject* object );
	// Push the appropriate object proxy for the object if such a proxy exists
	// and return YES. If it doesn't exist, leave the stack unchanged and return
	// NO.

WF_API NSObject* kslobjc_toNSObject( lua_State* L, int index, Class optClass );
	// Convert an item on the stack into an NSObject*. Optionally checks
	// the class. Returns nil if the object is inappropriate. Protects against
	// losing the Lua reference by doing a retain/autorelease (but that is
	// expensive and useless if you don't have an autorelease pool).
	
WF_API NSObject* kslobjc_toTempNSObject( lua_State* L, int index, Class optClass );
	// Like kslobjc_toNSObject but does not do the retain/autorelease. Use this
	// if you won't be mucking with the references from the Lua stack or
	// otherwise closing the Lua proxy.
	
WF_API void kslobjc_setNSObjectProxyExtraSize( lua_State* L, int index, size_t extraSize );
	// Object proxies can have an associated extra size that is used for
	// driving the GC forward. For example, a bitmap image should account for
	// the size of the bitmap. This only really matters when the extra size is
	// significant. For small objects, we don't need to do this accounting.
	
WF_API void kslobjc_closeNSObjectProxy( lua_State* L, int index );
	// Close the proxy object at the given index. Does nothing if the value is
	// nil. Logs an error if it is not a proxy. This will not throw exceptions
	// unless memory is really tight or object deallocation causes an error
	// to get thrown. Implemented by sending the close message to the proxy, so
	// it closes in exactly the same way as from the Lua side.
	
//==============================================================================

WF_API void kslobjc_pushCleansedSelector( lua_State* L, SEL selector );
	// Push a version of the selector "cleansed" for use on the Lua side.

WF_API void kslobjc_setDelegate( lua_State* L, int proxyIndex, BOOL strong );
	// Set the item at the top of the stack as the delegate for the proxy at
	// the given index. The delegate can be stored either weakly or strongly.
	// Generally, weak is a better choice because it avoids having the delegate
	// get resurrected by proxy finalization.
	
WF_API BOOL kslobjc_getDelegate( lua_State* L, int proxyIndex );
	// Get the delegate for the proxy. If the proxy lacks a delegate, return NO
	// and leave the stack unchanged.

WF_API BOOL kslobjc_prepDelegateSend( lua_State* L, NSObject* object, SEL selector );
	// Push the object's proxy if it exists. Get its environment. If the
	// environment is not the standard environment, then get the the implementation
	// for the selector if it exists. If all of this works, then leave the stack as
	// ( method env ) and return YES. If this fails at any point, leave the stack
	// unchanged and return NO. The selector will be "cleansed" to make it Lua
	// friendly.
	
//==============================================================================

#endif
