/*******************************************************************************

KSMacros.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2010 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

********************************************************************************

Utility macros for common operations.

*******************************************************************************/

#ifndef KSMacros_h
#define KSMacros_h

#include <stdlib.h>


//==============================================================================

#ifndef WIN_ENV
#pragma mark C code manipulations
#endif

//------------------------------------------------------------------------------

// Wrap multiple statements into a single statement block. Note that the
// expectation is that the code invoking this will be providing semicolon
// termination for its use but that the statements within will be complete
// including the terminating semicolon.

#define KSStatementBlock( statements ) do { statements } while( 0 )

//------------------------------------------------------------------------------

// An always-on version of assert(). Adapted from the generic case of assert.h.
// Abort the app with debug info if condition x is not met.

#define KSVerify( x ) \
	( (void) ( ( x ) ? 0 : KSVerify_imp( #x, __FILE__, __LINE__ ) ) )
#define KSVerify_imp( x, file, line ) \
    ( (void) printf( "%s:%u: failed assertion \"%s\"\n", file, line, x ), abort() )


//==============================================================================

#ifndef WIN_ENV
#pragma mark Object Lifetime Maintenance
#endif

//------------------------------------------------------------------------------

// Send release to a field (or less commonly a variable) and then clear it to
// nil. Useful in dealloc methods.

#define KSReleaseAndClear( fieldName ) KSStatementBlock( [ fieldName release ]; fieldName = nil; )

//------------------------------------------------------------------------------

// Update a retained field with a new value. This assumes the value will be
// changing but is safe if it doesn't. It just incurs unnecessary retain
// release traffic if it doesn't change. This does not send out any notifications
// for changes.

#define KSUpdateRetained( fieldName, newValue ) \
	KSStatementBlock( [ newValue retain ]; [ fieldName release ]; fieldName = newValue; )

//------------------------------------------------------------------------------

#define KSSupportsOrientationImp( orientation ) \
	KSStatementBlock( \
		if ( [ [ UIDevice currentDevice ] userInterfaceIdiom ] != UIUserInterfaceIdiomPad ) { \
			return orientation == UIInterfaceOrientationPortrait;\
		}\
		return YES; )\

#endif

//==============================================================================
/*
	If we are compiling under Objective-C we may want to wrap handler calls
	with auto release pools to avoid having auto-released objects build up.
	This increases the costs of handler calls but it avoids memory problems
	when working with Objective-C APIs. Be careful about foreign exceptions
	since it isn't clear how robust @autoreleasepool is to these.
*/

#if __OBJC__

#define KS_AUTO_RELEASE_POOL_BEGIN @autoreleasepool {
#define KS_AUTO_RELEASE_POOL_END }

#else

#define KS_AUTO_RELEASE_POOL_BEGIN {
#define KS_AUTO_RELEASE_POOL_END }

#endif
	
//==============================================================================

#ifndef WIN_ENV
#pragma mark Dispatch queue checks
#endif

#define ASSERT_MAIN_DISPATCH_QUEUE \
	assert( [ NSThread isMainThread ] )
	
#define ASSERT_NOT_MAIN_DISPATCH_QUEUE \
	assert( ![ NSThread isMainThread ] )

#define REEXECUTE_ON_MAIN_DISPATCH_QUEUE( code ) \
	if( ![ NSThread isMainThread ] ) { \
		dispatch_async( dispatch_get_main_queue(), ^{ code } ); \
		return; \
	}
	// Use this preamble in methods that can be called from any thread,
	// but want to requeue themselves on the main dispatcher.
