/*******************************************************************************

KSZString.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2011 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

********************************************************************************

Much of this code has been adapted from Lightroom (AgZString.h).

********************************************************************************

ZStrings are an Adobe-invented string format that aids in localization.

A ZString consists of a key-value pair where the key starts with three dollar
signs ("$") and ends in an equal sign.  The key serves to identify the usage
of the string in the code and to uniquify the string from other strings in
the application.  It is in the form of a "/"-separated path (with a leading
slash).  For example:
	"$$$/ZString/Subscope/Example=This is an example"

Because of their distinctive format, ZStrings can be "harvested" from code
by a tool which generates a ZString file.  This file can then be localized
by altering the value part of the strings, and when it is used by the
application, the localized values override the default values (i.e. English)
coded into the program for each key. The translated ZString file is the
translation map.

ZStrings can be converted to localized form using:

	KSZString.fromZString "$$$/ZString/Subscope/Example=This is an example"
	
This yields "This is an example" (or whatever the appropriate localized text
is).

ZStrings in code should consist entirely of low-ASCII characters.  The key
should only contain characters in the set "a-zA-Z0-9/".  The value can
contain any low-ASCII character.

ZStrings support a mechanism known as hat-substitution.  This is used to
put some common non-low-ASCII characters into the strings.  For example,
the hat sequence ^C will include the Copyright symbol in that location in
the resultant string.  The general purpose ^U can be used to encode any
arbitrary Unicode character given its Unicode code point in the form ^U+1234.
A hat followed by a number is a "replacement point".  It allows for the
sequence to be replaced by some other string programmatically.

	^r			ControlCarriageReturn
	^n			ControlLineFeed

	^B			Bullet
	^C			CopyrightSign
	^D			DegreeSign
	^I			Increment
	^R			RegisteredSign
	^S			NArySummation
	^T			TradeMarkSign
	^!			NotSign

	^{			LeftSingleQuotationMark
	^}			RightSingleQuotationMark
	^[			LeftDoubleQuotationMark
	^]			RightDoubleQuotationMark
	^'			RightSingleQuotationMark (curly apostrophe)
	^.			Ellipsis ("..." character)
		NOTE: ^. is deprecated and should not be used.  Use "..." instead.

	^e			LatinSmallLetterEWithAcute
	^E			LatinSmallLetterEWithCircumflex
	
	^d			GreekCapitalLetterDelta
	
	^L			\
	^V			|
	^#			CommandKey
	^`			`
	^^			^

	^0 - ^9		Replacement point for "replaceHat" functions

	^U+1234		Unicode code point U+1234

*******************************************************************************/


//==============================================================================

#import <Foundation/Foundation.h>

//------------------------------------------------------------------------------

#pragma mark -
#pragma mark KSZString functions

//------------------------------------------------------------------------------

NSString* NSString_fromZString( const char* inStringKey );
	/**< Given a string in ZString $$$key=default format (see above), return the
		 properly localized string. */

//------------------------------------------------------------------------------

BOOL NSString_looksLikeZString( const char* inString );
	/**< Return true if the string looks like a ZString (i.e. it starts with the
		 three dollar signs, etc.). */

//------------------------------------------------------------------------------

NSString* NSString_replaceHatWithString( NSString* inString,
										 int inWhichHatSequence, NSString* inReplacement );
	/**< Given a string that may contain replacement points identified by
		 ^0 .. ^9, replace any occurrences of replacement point whichHat with the
		 replacement string. */

//------------------------------------------------------------------------------

NSString* NSString_replaceHatWithInteger( NSString* inString,
												   int inWhichHatSequence, int inReplacement );
	/**< Given a string that may contain replacement points identified by
		 ^0 .. ^9, replace any occurrences of replacement point whichHat with the
		 replacement value (converted to string). */

//------------------------------------------------------------------------------

NSString* NSString_getISOLanguageCode( void );
	/**< return the ISO 639-1 language code that the application is currently running as
		 Example: "en", "ja", "fr", etc.. */

//------------------------------------------------------------------------------

NSString* NSString_escapeAmpersands( NSString* inString );
NSString* NSString_stripAccelerator( NSString* inString );

//------------------------------------------------------------------------------

#pragma mark -
#pragma mark Objective-C/NSString support

//------------------------------------------------------------------------------

//==============================================================================
// Category on NSString to support building from ZString key:

@interface NSString (KSZString)

+ (NSString*) stringWithZString: (const char*) inStringKey;

+ (NSString*) stringWithZString: (const char*) inStringKey
				and0Replacement: (NSString*) in0Replacement;

+ (NSString*) stringWithZString: (const char*) inStringKey
				andReplacements: (NSString*) inReplacement, ...;

+ (BOOL) looksLikeZString: (const char*) inString;

- (NSString*) localizePossibleZString;

+ (NSString*) getISOLanguageCode;

- (NSString*) stringWithStrippedAccelerator;
- (NSString*) stringWithEscapedAmpersands;

@end

//==============================================================================
// Category on NSMutableString to support building from ZString key and
// doing hat sequence (^0, ^1, ...) replacement:

@interface NSMutableString (KSZString)

+ (NSMutableString*) stringWithZString: (const char*) inStringKey;

- (NSMutableString*) replaceHat: (int) inWhichHatSequence withString: (NSString*) inReplacement;
- (NSMutableString*) replaceHat: (int) inWhichHatSequence withInteger: (int) inReplacement;

@end


//------------------------------------------------------------------------------

#pragma mark -
#pragma mark ZSTR

//------------------------------------------------------------------------------

//==============================================================================
//	This macro makes a ZString from a constant string literal in code.  The
//	design of the macro prevents calling it with a variable so that it could
//	be optimized to not copy the characters (in cases where this is possible).

#define ZSTR( x ) NSString_fromZString("" x "")
