/*******************************************************************************

WFLuaMessagingProcessor.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2012 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

#ifndef WFLuaMessagingProcessor_h
#define WFLuaMessagingProcessor_h

	// wichitafoundation/threading
#include <WFCore/WFMessaging.h>

#include <WFCore/WFApiDefs.h>

	// Lua 5.1
#include <AgKernel/lua.h>


//==============================================================================

WF_API
WFMessaging_ProcessorRef WFLuaMessagingProcessor_getCurrentProcessor( lua_State* L );
	/**<
		\brief Return the current message processor.
	*/

//------------------------------------------------------------------------------

WF_API
bool WFLuaMessagingProcessor_scheduleFunctionInLuaState(
									WFMessaging_ProcessorRef processor,
									WFMessaging_EnqueueOrder enqueueOrder,
									lua_CFunction func );
	/**<
		\brief Schedule a function to run in a Lua state on this messaging processor.
		
		This API does not provide any mechanism for prioritizing or canceling
		requests, nor does it provide any mechanism for the caller to receive
		results.
		
		\warning The effect of calling this when \c processor is not a Lua-provisioned
		message processor are undefined.
		
		\threadsafety This function may be called from any thread.
		
		\param processor the worker processor (thread) that is expected
			to respond to the request. Must be a Lua-provisioned message processor.
		\param enqueueOrder the position to use when enqueueing the message
		\param func the function to be called
		
		\return \c true if this message was successfully enqueued for processing;
			\c false if not. (A \c false return might occur, for example, if
			the message handler for this processor has been terminated.)
	*/

//------------------------------------------------------------------------------

WF_API
bool WFLuaMessagingProcessor_scheduleFunctionWithLightUserdataInLuaState(
									WFMessaging_ProcessorRef processor,
									WFMessaging_EnqueueOrder enqueueOrder,
									lua_CFunction func,
									void* ud );
	/**<
		\brief Schedule a function to run in a Lua state on this messaging processor.
		
		This API does not provide any mechanism for prioritizing or canceling
		requests, nor does it provide any mechanism for the caller to receive
		results.
		
		\warning The effect of calling this when \c processor is not a Lua-provisioned
		message processor are undefined.
		
		\threadsafety This function may be called from any thread.
		
		\param processor the worker processor (thread) that is expected
			to respond to the request. Must be a Lua-provisioned message processor.
		\param enqueueOrder the position to use when enqueueing the message
		\param func the function to be called
		\param ud the argument to the function to be passed as light userdata
		
		\return \c true if this message was successfully enqueued for processing;
			\c false if not. (A \c false return might occur, for example, if
			the message handler for this processor has been terminated.)
	*/

//------------------------------------------------------------------------------

WF_API
bool WFLuaMessagingProcessor_scheduleFunctionWithDataInLuaState(
									WFMessaging_ProcessorRef processor,
									WFMessaging_EnqueueOrder enqueueOrder,
									lua_CFunction func,
									WFMessaging_DataRef dataRef );
	/**<
		\brief Schedule a function to run in a Lua state on this messaging processor.
		
		This API does not provide any mechanism for prioritizing or canceling
		requests, nor does it provide any mechanism for the caller to receive
		results.
		
		\warning The effect of calling this when \c processor is not a Lua-provisioned
		message processor are undefined.
		
		\threadsafety This function may be called from any thread.
		
		\param processor the worker processor (thread) that is expected
			to respond to the request. Must be a Lua-provisioned message processor.
		\param enqueueOrder the position to use when enqueueing the message
		\param func the function to be called
		\param dataRef the argument to the function to be passed as KSData
		
		\return \c true if this message was successfully enqueued for processing;
			\c false if not. (A \c false return might occur, for example, if
			the message handler for this processor has been terminated.)
	*/

//------------------------------------------------------------------------------

WF_API
bool WFLuaMessagingProcessor_scheduleFunctionWithCleanupInLuaState(
									WFMessaging_ProcessorRef processor,
									WFMessaging_EnqueueOrder enqueueOrder,
									lua_CFunction func,
									void* data,
									void (*cleanupFunc)( void* data ) );
	/**<
		\brief Schedule a function to run in a Lua state on this messaging processor.
		
		This API does not provide any mechanism for prioritizing or canceling
		requests, nor does it provide any mechanism for the caller to receive
		results.
		
		\warning The effect of calling this when \c processor is not a Lua-provisioned
		message processor are undefined.
		
		\threadsafety This function may be called from any thread.
		
		\param processor the worker processor (thread) that is expected
			to respond to the request. Must be a Lua-provisioned message processor.
		\param enqueueOrder the position to use when enqueueing the message
		\param func the function to be called
		\param data arbitrary data pointer. If not \c NULL, this pointer will be
			passed as a light userdata item at position 1 on the Lua stack.
		\param cleanupFunc this function will be called after the Lua function
			terminates. It will also be called if the message fails to enqueue.
			The \c data pointer will be passed through. May be \c NULL, in which
			case no cleanup action will take place.
		
		\return \c true if this message was successfully enqueued for processing;
			\c false if not. (A \c false return might occur, for example, if
			the message handler for this processor has been terminated.)
	*/

//------------------------------------------------------------------------------

WF_API
void WFLuaMessagingProcessor_DEBUG_logMemoryAtStep( lua_State* L,
												    const char* stepName,
												    int stepDepthDelta );
	/**<
		\brief (for debugging purposes only) Log the amount of memory used at
			this stage of spinning up the Lua state.
		
		This function should have no side effects other than console log output.
		In some builds, it may have no effect at all.
		
		\param L the Lua state
		\param stepName name of the step being executed
		\param stepDepthDelta
			<ul>
				<li>+1 if entering a new step that may contain other steps</li>
				<li>0 if recording a step just completed</li>
				<li>-1 if exiting a step (must balance an earlier +1)</li>
			</ul>
	*/

//------------------------------------------------------------------------------

WF_API
void WFLuaMessagingProcessor_requestIdleGC( lua_State* L );
	/**<
		\brief Request that the processor correspondinf to this state perform
		idle time GC.
	*/
	
//------------------------------------------------------------------------------

WF_API
WFMessaging_Selector WFMessaging_Selector_fromLua( lua_State* L, int index );
    /**<
        \brief Convert a string at index in Lua state L to a messaging selector.
     
        \param L the Lua state
        \param index the index of the string in the Lua state
     
        \return the interned selector
    */

//------------------------------------------------------------------------------

WF_API
void WFMessaging_Selector_pushToLua( WFMessaging_Selector selector, lua_State* L );
    /**<
        \brief Push a selector to Lua.
     
        \param L the Lua state
        \param selector the selector to push (null results in pushing nil)
    */

//------------------------------------------------------------------------------

WF_API
WFMessaging_ExternalMessageRef WFMessaging_ExternalMessage_createLuaMessage(
			WFMessaging_ProcessorRef luaProcessorRef,
			WFMessaging_EnqueueOrder enqueueOrder,
			lua_CFunction fn,
			void* data );
    /**<
        \brief Create an external message for repeatedly invoking a Lua function
		in a Lua messaging processor. This does not keep the message processor
		alive.
     
        \param luaProcessorRef the processor (must be a Lua message processor)
		\param enqueueOrder the position to use when enqueueing the message
        \param fn the function
		\param data a single parameter for the function passed as light userdata
     
        \return the external message object
    */

//------------------------------------------------------------------------------

#endif
