/*******************************************************************************

WFLuaRCObject.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2010 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

#ifndef WFLuaRCObjectBridge_h
#define WFLuaRCObjectBridge_h

#include <AgKernel/lua.h>
#include <AgKernel/lauxlib.h>

#include "WFApiDefs.h"

#if MAC_ENV || IOS_ENV || ANDROID_ENV

#include <stdbool.h>

#endif


//==============================================================================

// Declarations for IUnknown.

#if MAC_ENV || IOS_ENV || ANDROID_ENV

typedef struct IUnknown_vtbl* IUnknown;

struct IUnknown_vtbl {
	void (*AddRef)( IUnknown* );
	void (*Release)( IUnknown* );
};

#define DLL_EXPORT

#elif WIN_ENV || WINRT_ENV

#include <unknwn.h>

#endif


//==============================================================================

// The declaration for the object class itself.

typedef void* WFLuaRCObjectPtr;

// The description of the object class. (Defined below.)

typedef const struct WFLuaRCClassDesc *WFLuaRCClassDescPtr;


//==============================================================================

// We describe reference counting using a single function that takes a delta
// count value. This values will always be either +1 or -1.

typedef void (*WFLuaRCObjectRetentionFunc)(
			WFLuaRCClassDescPtr classDesc, WFLuaRCObjectPtr object, int delta );


// We provide some standard functions.

WF_API void WFLuaRCObjectRetention_COM(
			WFLuaRCClassDescPtr classDesc, WFLuaRCObjectPtr object, int delta );
	// COM objects managed via AddRef and Release. The object pointer is expected
	// to point to a COM interface.

#if MAC_ENV || IOS_ENV

WF_API void WFLuaRCObjectRetention_ObjC(
			WFLuaRCClassDescPtr classDesc, WFLuaRCObjectPtr object, int delta );
	// Objective-C objects managed via retain and release.
			
WF_API void WFLuaRCObjectRetention_CF(
			WFLuaRCClassDescPtr classDesc, WFLuaRCObjectPtr object, int delta );
	// CoreFoundation objects managed via CFRetain and CFRelease.

#endif


//==============================================================================

// We describe COM classes with a name and a set of methods. The methods are
// added to the metatable if they start with __ (you get what you deserve if
// you add __index since this leads to the methods table) or the methods
// table otherwise. The methods table should be terminared with NULL, NULL.

// UNIMPLEMENTED EXTENSIONS: We could use special names or constructs to do
// things like run code at class load time in a universe or to provide an
// opportunity to add Lua methods.

typedef struct WFLuaRCClassDesc {
	const char* className;
	WFLuaRCObjectRetentionFunc retention;
	const luaL_reg *methods;
} WFLuaRCClassDesc;


//==============================================================================

// Pushing and retrieving reference-counted objects.

WF_API void wflrco_pushRCObjectProxy(
				lua_State* L,
				WFLuaRCClassDescPtr classDesc,
				WFLuaRCObjectPtr object,
				size_t gcPressure );
	// Push the appropriate object proxy for the object.
	
WF_API int wflrco_pushRCObjectProxyIfExists(
					lua_State* L,
					WFLuaRCClassDescPtr classDesc,
					WFLuaRCObjectPtr object );
	// Push the appropriate object proxy for the object if such a proxy exists
	// and return 1. If it doesn't exist, leave the stack unchanged and return
	// 0.
	

WF_API WFLuaRCObjectPtr wflrco_pushRCObjectProxyAssumingOwnership(
				lua_State* L,
				WFLuaRCClassDescPtr classDesc,
				WFLuaRCObjectPtr object,
				size_t gcPressure );
	// Push an object proxy for the object taking over the release obligation.
	// Returns the object. Pushes nil if given NULL.
	
WF_API WFLuaRCObjectPtr wflrco_toTempRCObject(
					lua_State* L,
					int index,
					WFLuaRCClassDescPtr classDesc );
	// Convert an item on the stack into a WFLuaRCObjectPtr. Optionally checks
	// the interface marker from the proxy against the supplied interface
	// marker. Returns NULL if the object is inappropriate. If the stack
	// entry will go away, the caller needs to do an appropriate AddRef/retain
	// since this routine does not do so.
	
WF_API WFLuaRCObjectPtr wflrco_checkTempRCObject(
					lua_State* L,
					int index,
					WFLuaRCClassDescPtr classDesc );
	// This routine is only intended for use with function arguments. It
	// will throw a Lua exception if the object at the indicated location
	// is not of the expected type.
	
WF_API int wflrco_closeRCObject(
					lua_State* L,
					int index,
					WFLuaRCClassDescPtr classDesc );
	// If this stack slot contains an RCObject proxy for the given class,
	// then close the proxy and return 1. Otherwise, return 0.

//==============================================================================

// We can package up all of the information about an object in an object descriptor.

typedef struct WFLuaRCObjectDesc {
	WFLuaRCClassDescPtr classDesc;
	WFLuaRCObjectPtr object;
	size_t gcPressure;
} WFLuaRCObjectDesc;

WF_API bool wflrco_isTransitableRCObject(
					lua_State* L,
					int index,
					WFLuaRCObjectDesc* outObjectDesc );
	// Determine whether the object at the given index on the stack is an
	// RCObject and if so fill in the appropriate object descriptor.
	
WF_API void wflrco_pushRCObjectProxyForObjectDesc(
				lua_State* L,
				const WFLuaRCObjectDesc* objectDesc );
	// Push an object for a given object descriptor.

WF_API void wflrco_retainRCObjectForObjectDesc(
					const WFLuaRCObjectDesc* objectDesc );
	// Increment the reference count for an object described by an object
	// descriptor.

WF_API void wflrco_releaseRCObjectForObjectDesc(
					const WFLuaRCObjectDesc* objectDesc );
	// Decrement the reference count for an object described by an object
	// descriptor.
	

//==============================================================================

WF_API void wflrco_setRCObjectLink(
					lua_State* L,
					int objectIndex );
	// Set a link on an object using the top two values on the stack (key, value).
	// Like lua_settable but for object proxies. This throws a Lua exception if
	// used with something that isn't an object proxy.
	
WF_API void wflrco_getRCObjectLink(
					lua_State* L,
					int objectIndex );
	// Get a link on an object using the key value from the stack. Like lua_gettable
	// but for object proxies. This throws a Lua exception if used with something
	// that isn't an object proxy.
	
WF_API void wflrco_setRCObjectLinkAtName(
					lua_State* L,
					int objectIndex,
					const char* fieldName );
	// Set a link on an object using a named field. Like lua_setfield but for
	// object proxies. This throws a Lua exception if used with something that
	// isn't an object proxy.

WF_API void wflrco_getRCObjectLinkAtName(
					lua_State* L,
					int objectIndex,
					const char* fieldName );
	// Get a link on an object using a named field. Like lua_getfield but for
	// object proxies. This throws a Lua exception if used with something that
	// isn't an object proxy.

WF_API void wflrco_setRCObjectLinkAtIndex(
					lua_State* L,
					int objectIndex,
					int index );
	// Set a link on an object using an integer index. Like lua_rawseti but for
	// object proxies. This throws a Lua exception if used with something that
	// isn't an object proxy.

WF_API void wflrco_getRCObjectLinkAtIndex(
					lua_State* L,
					int objectIndex,
					int index );
	// Get a link on an object using an integer index. Like lua_rawseti but for
	// object proxies. This throws a Lua exception if used with something that
	// isn't an object proxy.
	

//==============================================================================

#endif
