/*******************************************************************************

WFMessaging.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2011 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

/*!
	\file WFMessaging.h

	\brief \c WFMessaging provides the core mechanism for inter-thread messaging.
*/

#ifndef WFMessaging_h
#define WFMessaging_h

#if IOS_ENV || MAC_ENV || ANDROID_ENV
#include <stdbool.h>
#endif

#include <stddef.h>

#include "WFMessagingData.h"
#include "WFMessagingSelectors.h"
#include "WFMessagingThreadPriorities.h"
#include "WFApiDefs.h"

//==============================================================================

#pragma mark -
#pragma mark typedefs

//------------------------------------------------------------------------------

typedef struct WFMessaging_Processor_t* WFMessaging_ProcessorRef;
	/**<
		\brief The message processor object associates a thread,
		a message queue, and a messaging handler.
	*/

//------------------------------------------------------------------------------

typedef struct WFMessaging_Request_t* WFMessaging_RequestRef;
	/**<
		\brief The messaging request object represents a single request from a
		client thread to a worker thread. The existence of the object does not
		guarantee that the client thread is active.

		\todo Does it keep the worker thread alive?
	*/

//------------------------------------------------------------------------------

typedef int WFMessaging_Priority;
	/**<
		\brief The priority of a request, which can be used to signal its relative
		importance.

		The default priority is zero. (See \c WFMessaging_Priority_Default.)
		Use positive integers to signal higher-than-default priority, and
		negative integers to signal lower-than-default priority.
		Do not use priority values of -10000 or smaller. (See \c WFMessaging_Priority_Cancel.)

		A message processor is not required to honor request priorities.
	*/

#define WFMessaging_Priority_Default ( (WFMessaging_Priority) 0 )
	/**< \brief Default request priority. */

#define WFMessaging_Priority_Cancel ( (WFMessaging_Priority) -10000 )
	/**< \brief Reserved priority value used internally to cancel a request. */

//------------------------------------------------------------------------------
/*!
	\brief Status of a message request.
*/
typedef enum WFMessaging_RequestStatus {
	wfm_rsNotDone, 	/**< Request is still expected to return further results. */
	wfm_rsDone,		/**< Request has delivered its last result. */
	wfm_rsOrphaned,	/**< Request lost the worker side without delivering a final result. */
} WFMessaging_RequestStatus;

//------------------------------------------------------------------------------
/*!
	\brief Where to enqueue a message request.
*/
typedef enum WFMessaging_EnqueueOrder {
	wfm_eoFIFO, 	/**< Enqueue for FIFO execution. Use when sequence order matters. */
	wfm_eoLIFO,		/**< Enqueue for LIFO execution. Use when executing soon matters. */
	wfm_eoFast = wfm_eoLIFO,
	wfm_eoInOrder = wfm_eoFIFO
} WFMessaging_EnqueueOrder;


//==============================================================================

#pragma mark -
#pragma mark WFMessaging_Handler

//------------------------------------------------------------------------------

typedef struct WFMessaging_Handler WFMessaging_Handler;
	/**<
		\brief The message handler object encapsulates a set of handler functions
		to process various types of requests.
	*/

//------------------------------------------------------------------------------
/*!
	\brief A set of functions that implement a \c WFMessaging_Handler.

	\threadsafety The message processing engine guarantees thread safety for each
	message handler instance. The engine never makes simultaneous calls
	from different threads to the same handler instance. This applies to all
	messages described in this interface.
*/
typedef struct WFMessaging_Handler_Procs {

	//--------------------------------------------------------------------------

	void (*initialize)( WFMessaging_Handler* self,
					    WFMessaging_ProcessorRef processorRef );
		/**<
			\brief Starts the messaging handler. The handler should complete whatever
			initialization it needs in order to receive other messages before
			returning from this call.

			This call is made on the thread belonging to the message processor.

			If the messaging handler encounters an error on startup and is unable
			to process messages, call \c WFMessaging_Processor_handlerDidFail(),
			which immediately terminates the messaging handler and replaces it
			with a special handler that responds with a predefined
			failure message to all requests.

			\param self The handler object that was	created using
				 the \c WFMessaging_HandlerFactory function.
			\param processorRef The messaging processor that owns this handler.

			\warning Do \b NOT retain this reference; that is, do not call either
				<code>WFMessaging_Processor_strongRetain( processorRef )</code> or
				<code>WFMessaging_Processor_weakRetain( processorRef )</code>.
				Doing so causes a memory leak. You should simply store the pointer
				reference.
		*/

	//--------------------------------------------------------------------------

	void (*handleExternalMessage)( WFMessaging_Handler* self,
								   WFMessaging_Selector selector,
								   WFMessaging_DataRef dataRef );
		/**<
			\brief Handles a fire-and-forget message from another thread.

			This type of message does not provide any mechanism for prioritizing or canceling
			the request, and does not communicate results.
			Typically used for system-wide broadcast notifications.

			\param self The handler object that was	created using
				 the \c WFMessaging_HandlerFactory function.
			\param selector The message ID.
			\param dataRef Additional developer-defined data associated with the request, or \c NULL.
		*/

	//--------------------------------------------------------------------------

	void (*handleRequest)( WFMessaging_Handler* self,
						   WFMessaging_RequestRef requestRef,
						   WFMessaging_Priority priority,
						   WFMessaging_Selector selector,
						   WFMessaging_DataRef dataRef );
		/**<
			\brief Handles a standard request from another thread.

			Called on the worker thread.

			\param self The handler object that was	created using
				 the \c WFMessaging_HandlerFactory function.
			\param requestRef A tracking object, used to communicate results back to the client thread
				  or for the client thread to signal priority changes or cancelation.

			\param selector The message ID.
			\param dataRef Additional developer-defined data associated with the request, or \c NULL.
		*/

	//--------------------------------------------------------------------------

	void (*handleResponse)( WFMessaging_Handler* self,
						    WFMessaging_RequestRef requestRef,
						    WFMessaging_RequestStatus requestStatus,
						    WFMessaging_Selector statuSelector,
						    WFMessaging_DataRef dataRef );
		/**<
			\brief Receives the response to a request initiated by this client thread.

			Called on the client thread.

			\param self The handler object that was	created using
				 the \c WFMessaging_HandlerFactory function.
			\param requestRef A request tracking object; matches the result from the
				call to \c WFMessaging_Processor_clientCreateAndEnqueueRequest()
				that initiated this request
			\param requestStatus A single request can result in any number of responses,
			depending on how the worker thread's messaging handler interprets
			the message. This status tells you what	to expect.

			\param statusSelector The message status, such as "data" or "error".
			\param dataRef Additional developer-defined data associated with the request, or \c NULL.
		*/

	//--------------------------------------------------------------------------

	void (*handlePriorityChange)( WFMessaging_Handler* self,
								  WFMessaging_RequestRef requestRef,
								  WFMessaging_Priority priority );
		/**<
			\brief Receives a change in priority for a request from the client
			thread.

			Called on the worker thread. Worker threads are encouraged, but not required,
			to use priority as a hint in scheduling requests.

			\param self The handler object that was	created using
				 the \c WFMessaging_HandlerFactory function.
			\param requestRef A request tracking object that matches this request to
				the \c handleRequest() call that sent it to this thread.
			\param priority A new priority set by the client for this request.
				To cancel the request, use the reserved priority value
				of \c WFMessaging_Priority_Cancel.
		*/

	//--------------------------------------------------------------------------

	void (*pulse)( WFMessaging_Handler* self );
		/**<
			\brief Responds to a call to \c WFMessaging_Processor_enqueuePulseAfterDelay().
			Called when the specified delay interval has elapsed,

 			\param self The handler object that was	created using
				 the \c WFMessaging_HandlerFactory function.
		*/

	//--------------------------------------------------------------------------

	void (*idle)( WFMessaging_Handler* self,
				  bool* repeat );
		/**<
			\brief Performs work when no other messages are waiting for this thread.

			Called when the queue of messages waiting to be
			delivered is empty, but only if there has been at least one call to
			\c WFMessaging_Processor_enqueueIdle since the message handler was
			created or last received an \c idle message.

			\param self The handler object that was	created using
				 the \c WFMessaging_HandlerFactory function.
			\param repeat When true, the handler expects to receive another idle event.
				Default is \c false, meaning no new idle event is received.
		*/

	//--------------------------------------------------------------------------

	void (*terminate)( WFMessaging_Handler* self );
		/**<
			\brief Stops receiving events.

			Called when the messaging handler is no longer needed and should stop doing any work.

			Called only if the initialization is successful, and is always followed immediately
			by the \c free() call.

			Once this call occurs, no other messages are sent to this message
			handler except for the \c free call.
		*/

	//--------------------------------------------------------------------------

	void (*free)( WFMessaging_Handler* self );
		/**<
			\brief Releases memory and other resources associated with this
			message handler.

			This is the last message to be received by any message handler.
		*/

	//--------------------------------------------------------------------------

} WFMessaging_Handler_Procs;

//------------------------------------------------------------------------------
/*!
	\brief Abstract interface for a handler that interprets messages from other threads and
	performs the work that was requested.

	Provide different implementations of this interface for different kinds of message handlers.

	\threadsafety The message processing engine never makes simultaneous calls
	from different threads to the same handler instance. This applies to all
	messages described in this interface.
*/
struct WFMessaging_Handler {
	WFMessaging_Handler_Procs* vt;
};


//==============================================================================

#pragma mark -
#pragma mark WFMessaging_Processor

//------------------------------------------------------------------------------

WF_API WFMessaging_ProcessorRef WFMessaging_Processor_create( const char* name,
															  WFMessaging_Handler* handler );
	/**<
		\brief Creates a new messaging processor.

		The message processor gains a strong reference as a result of this call.
		The caller has an obligation to release that reference by calling
		\c WFMessaging_Processor_strongRelease() when the message queue is no longer
		needed.

		This function calls the handler's \c initialize() function
		in the worker thread.

		\warning Do NOT pass the same handler object to multiple message processors.

		\threadsafety This function may be called from any thread.

		\param name A name for this messaging processor thread, which is used for debugging.
		\param handler The messaging handler for this processor. Must not be \c NULL.

		\return The new messaging processor object.
	*/

//------------------------------------------------------------------------------

WF_API WFMessaging_ProcessorRef WFMessaging_Processor_strongRetain(
					WFMessaging_ProcessorRef processorRef );
	/**<
		\brief Increments the strong reference count for message processor. A strong
		reference ensures that both the processor and its message handler are kept
		alive.

		\threadsafety This function may be called from any thread.

		\param processorRef The messaging processor object. If \c NULL, the
			call does nothing.

		\return The messaging processor object (useful for chaining).
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Processor_strongRelease( WFMessaging_ProcessorRef processorRef );
	/**<
		\brief Decrements the strong reference count for a messaging processor, and if the
		count falls to zero, frees the message handler. If the
		weak reference count has also fallen to zero, also frees the message
		processor itself.

		\warning Be careful to correctly match strong vs weak retains and releases.

		\threadsafety This function may be called from any thread.

		\param processorRef The messaging processor object. If \c NULL, the
			call does nothing.
	*/

//------------------------------------------------------------------------------

//
// Not implemented: WF_API void WFMessaging_Processor_strongReleaseAndClear( WFMessaging_ProcessorRef* processorRefPtr );
	/**<
		\brief Decrements the strong reference count for a message processor.
		If the strong reference count falls to zero, frees
		the message handler. If the weak reference count has also fallen to zero,
	    also frees the message processor itself.

		\warning Be careful to correctly match strong vs weak retains and releases.

		\threadsafety This function may be called from any thread.

		\param processorRefPtr The request to release. If \c NULL, this call has no effect.
			On exit, this pointer is set to \c NULL.
	*/

//------------------------------------------------------------------------------

WF_API WFMessaging_ProcessorRef WFMessaging_Processor_weakRetain(
					WFMessaging_ProcessorRef processorRef );
	/**<
		\brief Increments the weak reference count for a messaging processor. A weak
		reference ensures that the processor is kept alive, but does not ensure
		that its message handler is kept alive.

		\param processorRef The messaging processor to retain.  If \c NULL, the
			call does nothing.

		\threadsafety This function may be called from any thread.

		\return The messaging processor object (useful for chaining).
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Processor_weakRelease( WFMessaging_ProcessorRef processorRef );
	/**<
		\brief Decrements the weak reference count for a messaging processor. If the
		total reference count has fallen to zero, also frees the messaging processor.

		\warning Be careful to correctly match strong vs weak retains and releases.

		\threadsafety This function may be called from any thread.

		\param processorRef The messaging processor to release.  If \c NULL, the
			call does nothing.

	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Processor_weakReleaseAndClear( WFMessaging_ProcessorRef* processorRefPtr );
	/**<
		\brief Decrements the strong reference count for a messaging processor.
		If the total reference count has fallen to zero, also frees the messaging processor.

		\warning Be careful to correctly match strong vs weak retains and releases.

		\threadsafety This function may be called from any thread.

		\param processorRefPtr A pointer to the messaging processor to release. If \c NULL, the
			call does nothing. On exit, this pointer is set to \c NULL.
	*/

//------------------------------------------------------------------------------

WF_API WFMessaging_RequestRef WFMessaging_Processor_clientCreateAndEnqueueRequest(
					WFMessaging_ProcessorRef clientProcessorRef,
					WFMessaging_ProcessorRef workerProcessorRef,
					WFMessaging_Priority priority,
					WFMessaging_Selector selector,
					WFMessaging_DataRef dataRef );
	/**<
		\brief Creates a request message on behalf of a client message processor
		(client thread) and sends it to a worker processor (worker thread).

		The request starts with a single client reference as a result of this call.
		The caller is responsible for releasing that reference by calling
		\c WFMessaging_Request_clientRelease() when the client side of the request
		is no longer needed.

		This function triggers a call to the \c handleRequest() function in the worker thread.

		\threadsafety This function may be called from any thread.

		\param clientProcessorRef The client thread that is making
			the request. Must not be \c NULL.
		\param workerProcessorRef The worker thread that is expected
			to respond to the request. Must not be \c NULL.
		\param priority The initial priority for the request. Default is 0.
		\param selector The message ID for the request. Must not be \c NULL.
		\param dataRef Additional developer-defined context data associated with the request, or \c NULL.

		\return A request reference that tracks this request until it is complete.
	*/

//------------------------------------------------------------------------------

WF_API WFMessaging_RequestRef WFMessaging_Processor_clientCreateAndEnqueueRequest_v2(
					WFMessaging_ProcessorRef clientProcessorRef,
					WFMessaging_ProcessorRef workerProcessorRef,
					WFMessaging_Priority priority,
					bool useFastEnqueue,
					bool useFastInitialResponse,
					WFMessaging_Selector selector,
					WFMessaging_DataRef dataRef );
	/**<
		\brief Creates a request message on behalf of a client message processor
		(client thread) and sends it to a worker processor (worker thread).

		The request starts with a single client reference as a result of this call.
		The caller is responsible for releasing that reference by calling
		\c WFMessaging_Request_clientRelease() when the client side of the request
		is no longer needed.

		This function triggers a call to the \c handleRequest() function in the worker thread.

		\threadsafety This function may be called from any thread.

		\param clientProcessorRef The client thread that is making
			the request. Must not be \c NULL.
		\param workerProcessorRef The worker thread that is expected
			to respond to the request. Must not be \c NULL.
		\param priority The initial priority for the request. Default is 0.
		\param useFastEnqueue Enqueues the request at the front of the worker's queue.
		\param selector The message ID for the request. Must not be \c NULL.
		\param dataRef Additional developer-defined context data associated with the request, or \c NULL.

		\return A request reference that tracks this request until it is complete.
	*/

//------------------------------------------------------------------------------

typedef void (*WFMessaging_RequestCallback)(
					void* context,
					WFMessaging_RequestRef requestRef,
					WFMessaging_RequestStatus requestStatus,
					WFMessaging_Selector statusSelector,
					WFMessaging_DataRef dataRef );
	/**<
		\brief The callback function to pass to \c WFMessaging_Processor_externalClientCreateAndEnqueueRequest().

		 Implement a function to handle results from the worker thread.
		 This should generally do as little work as possible
		 in order to avoid blocking. Called either on the worker thread or in response
		 to a call to \c WFMessaging_Request_clientRelease() from the client thread.
		 Can be called before request itself has returned.

		\param context The caller-defined context value passed when the request was made.
		\param requestRef The request object returned when the request was made.
		\param requestStatus The worker thread's status for this request
		\param statusSelector The message status, such as "data" or "error".
		\param dataRef Additional developer-defined context data associated with the request, or \c NULL.
	*/

//------------------------------------------------------------------------------

WF_API WFMessaging_RequestRef WFMessaging_Processor_externalClientCreateAndEnqueueRequest(
					WFMessaging_RequestCallback callback,
					void* callbackContext,
					WFMessaging_ProcessorRef workerProcessorRef,
					WFMessaging_Priority priority,
					WFMessaging_Selector selector,
					WFMessaging_DataRef dataRef );
	/**<
		\brief Creates a request message on behalf of an external client thread
		(not a WF messaging processor) and sends it to the worker processor (worker
		thread).

		The request starts with a single client reference as a result of this call.
		The caller is responsible for releasing that reference by calling
		\c WFMessaging_Request_clientRelease() when the client side of the request
		is no longer needed.

		This message triggers a call to the \c handleRequest()
		function in the worker thread.

		This function guarantees that it will generate a call to the callback
		with some terminal request status.

		\threadsafety This function may be called from any thread.

		\param callback A function to handle results from the worker thread.
			Must not be \c NULL. This should generally do as little work as possible
			in order to avoid blocking. Called either on the worker thread or in response
			to a call to \c WFMessaging_Request_clientRelease() from the client thread.
			Can be called before this routine itself has returned.
		\param callbackContext Additional developer-defined context data to pass to the callback,
			or \c NULL.
		\param workerProcessorRef The worker thread that is expected
			to respond to the request. Must not be \c NULL.
		\param priority The initial priority for the request.
		\param selector The message ID for the request. Must not be \c NULL.
		\param dataRef Additional developer-defined context data associated with the request, or \c NULL.

		\return A request reference that tracks this request until it is complete.
	*/

//------------------------------------------------------------------------------

WF_API bool WFMessaging_Processor_enqueueExternalMessage(
					WFMessaging_ProcessorRef processorRef,
					WFMessaging_EnqueueOrder enqueueOrder,
					WFMessaging_Selector selector,
					WFMessaging_DataRef dataRef );
	/**<
		\brief Sends a fire-and-forget message to another thread.

		This type of message does not provide any mechanism for prioritizing or canceling
		the request, and does not communicate results.
		Typically used for system-wide broadcast notifications.

		This message triggers a call to the \c handleExternalMessage()
		function in the worker thread.

		\threadsafety This function may be called from any thread.

		\param processorRef The worker thread that is expected
			to respond to the request. Must not be \c NULL.
		\param enqueueOrder The execution-queue location for the message.
			Enqueue for FIFO execution 	when sequence order matters, and
			for LIFO execution when executing soon matters.
		\param selector The message ID.
		\param dataRef Additional developer-defined context data associated with the request,
			or \c NULL.

		\return \c true if this message was successfully enqueued for processing;
			\c false if not. (A \c false return might occur, for example, if
			the message handler for this processor has been terminated.)
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Processor_broadcastExternalMessageToAllProcessors(
					WFMessaging_EnqueueOrder enqueueOrder,
					WFMessaging_Selector selector,
					WFMessaging_DataRef dataRef );
	/**<
		\brief Send a fire-and-forget message to all threads currently in
		operation that have been created by  \c WFMessaging_Processor_createForId().

		This type of message does not provide any mechanism for prioritizing or canceling
		the request, and does not communicate results.
		Typically used for system-wide broadcast notifications.

		This message triggers a call to the \c handleExternalMessage()
		function in each worker thread.

		\threadsafety This function may be called from any thread.

		\param selector The message ID.
		\param dataRef Additional developer-defined context data associated with the request,
			or \c NULL.
		\param enqueueOrder The execution-queue location for the message.
			Enqueue for FIFO execution 	when sequence order matters, and
			for LIFO execution when executing soon matters.
	*/

//------------------------------------------------------------------------------

WF_API bool WFMessaging_Processor_maybeStillRunning( WFMessaging_ProcessorRef processorRef );
	/**<
		\brief Reports whether a message processor might still accept and
		handle messages.

		This is not a definitive test, because the processor could be marked
		for termination but not yet terminated.

		\threadsafety This function may be called from any thread.

		\return Returns \c true if the message processor has a currently-valid message handler.
	*/

//------------------------------------------------------------------------------

WF_API bool WFMessaging_Processor_didTerminate( WFMessaging_ProcessorRef processorRef );
	/**<
		\brief Reports whether a message processor's message handler has
		completely shut down.

		Never returns a false positive result, but can return a false negative result
		if processor is in the process of terminating (which could complete before this call
		completes).

		\threadsafety This function may be called from any thread.

		\return Returns \c true if the message processor has terminated.
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Processor_enqueueIdle( WFMessaging_ProcessorRef processorRef );
	/**<
		\brief Requests an \c idle message when the message queue becomes empty.

		If you call this multiple times for the same processor, only one \c idle message is sent.
		This flag resets just before the \c idle message is called, so a message handler can requeue
		itself for \c idle processing during the \c idle call. If no other messages
		have been enqueued during the \c idle call, a second \c idle message is
		sent immediately. If other messages do arrive, then the second \c idle
		message occurs after those messages are handled.

		\threadsafety This function should only be called from within the message
		handler.

		\param processorRef The message handler's message processor.
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Processor_enqueuePulseAfterDelay( WFMessaging_ProcessorRef processorRef,
														  double delay );
	/**<
		\brief Requests a \c pulse message after a specified delay.

		There is no mechanism to cancel execution of this function once it has
		been scheduled. Multiple requests to the same processor send multiple messages.

		Pending pulse requests do not prevent the message handler from being terminated
		if the message processor loses its strong references.

		\threadsafety This function should only be called from within the message
		handler.

		\param processorRef The message handler's message processor.
		\param delay The time to wait, in seconds, before scheduling the pulse
			message. Must be a positive (non-zero) number.
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Processor_setPriority( WFMessaging_ProcessorRef processorRef,
											   WFMessaging_ThreadPriority priority );
	/**<
		\brief Changes the priority for this message processor.

		\threadsafety This function may be called from any thread.

		\param processorRef The processor to release. If \c NULL, the call has no effect.
		\param priority The new thread priority.
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Processor_handlerDidFail( WFMessaging_ProcessorRef processorRef,
												  WFMessaging_Selector failureStatusSelector,
												  WFMessaging_DataRef failureMessage );
	/**<
		\brief Terminates the message handler; all subsequent requests fail.

		Signals a major failure of the message handler. The handler is terminated
		by calling its \c terminate() handler (if it initialized successfully) and then
		freed by calling its \c free() handler.

		If a failure message is supplied, this processor retains the data
		reference and uses it in reply to any subsequent requests.

		If neither a status selector nor a failure message is supplied, this
		processor ignores all subsequent requests.

		\threadsafety This function should only be called from within the message
		handler.

		\param processorRef The message handler's message processor.
		\param failureStatusSelector A status-selector constant to be handed back in
			response to all subsequent requests to this message processor, or \c NULL.
		\param failureMessage A message to be handed back in response to
			all subsequent requests to this message processor, or \c NULL.
	*/


//==============================================================================

#pragma mark -
#pragma mark message handler factory

//------------------------------------------------------------------------------

typedef WFMessaging_Handler* (*WFMessaging_HandlerFactory)( const char* selector );
	/**<
		\brief Creates a messaging handler object for a developer-defined selector.

		The implementation must ensure that no more than one messaging processor
		and handler for a given selector is alive at any one time.

		\threadsafety This function may be called from any thread.

		\param selector The name for the message the new object handles. Must not
			be \c NULL.	Should be a unique string, such as a source file name.

		\return The new message handler object, or \c NULL if the request is not valid
			for this factory.
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_addHandlerFactory( WFMessaging_HandlerFactory factory );
	/**<
		\brief Registers a new factory for creating message processors.

		Factory functions are used by \c WFMessaging_Queue_createQueueForId() to
		create new message processors when needed.

		\threadsafety This function may be called from any thread.

		\param factory The new factory function.
	*/

//------------------------------------------------------------------------------

WF_API WFMessaging_ProcessorRef WFMessaging_Processor_createForId( const char* selector );
	/**<
		\brief Finds or creates a message processor for a given selector.

		If a message processor for this selector already exists and is still running,
		a new reference to the existing processor is returned.

		If not, this function iterates through all of the message handler factories
		that have been registered via \c WFMessaging_addHandlerFactory(), in order
		of registration, until one of them returns a non-\c NULL result.

	 	The message processor gains a strong reference as a result of this call.
		The caller is responsible for releasing that reference by calling
		\c WFMessaging_Processor_strongRelease() when the message queue is no longer
		needed.

		\threadsafety This function may be called from any thread.

		\param selector The developer-defined selector string. If not supplied, the call returns \c NULL.

		\return A reference to the new or existing processor, or \c NULL if no processor
			could be created
	*/


//==============================================================================

#pragma mark -
#pragma mark WFMessaging_Request

//------------------------------------------------------------------------------

WF_API WFMessaging_Selector WFMessaging_Request_getSelector( WFMessaging_RequestRef requestRef );
	/**<
		\brief Retrieves the selector for a message.

		\threadsafety Should only be called from the client or worker thread.

		\param requestRef The message request. If not supplied, the call returns \c NULL.

		\return The selector provided when this message was initially requested,
			or \c NULL if no message request is supplied.
	*/

//------------------------------------------------------------------------------

WF_API WFMessaging_RequestRef WFMessaging_Request_clientRetain( WFMessaging_RequestRef requestRef );
	/**<
		\brief Increments the client reference count for a request.

		Client and worker thread references are counted separately.

		\threadsafety Should only be called from the client thread that originated this request.

		\param requestRef The request to retain. If \c NULL, this call has no effect.

		\return The request reference (useful for chaining).
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Request_clientRelease( WFMessaging_RequestRef requestRef );
	/**<
		\brief Decrements the client reference count for a request. If the
		client reference count falls to zero, cancels the request. If the worker
		reference count has also fallen to zero, frees the request object.

		\warning Be careful to correctly match client vs worker retains and releases.

		\threadsafety Should only be called from the client thread that originated this request).

		\param requestRef The request to release. If \c NULL, this call has no effect.
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Request_clientReleaseAndClear( WFMessaging_RequestRef* requestRefPtr );
	/**<
		\brief Decrements the client reference count for a request processor.

		If the client reference count falls to zero, cancels
		the request. If the worker reference count has also fallen to zero, frees
		the request object.

		\warning Be careful to correctly match client vs worker retains and releases.

		\threadsafety Should only be called from the client thread that originated this request.

		\param requestRefPtr The request to release. If \c NULL, this call has no effect.
			On exit, this pointer is set to \c NULL.
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Request_clientSetPriority( WFMessaging_RequestRef requestRef,
												   WFMessaging_Priority priority );
	/**<
		\brief Changes the priority for a request by calling the
		\c handlePriorityChange function in the worker thread's message handler.

		If the request has been canceled, the call has no effect.
 		\threadsafety Should only be called from the client thread that originated this request.

		\param requestRef The request to update. If \c NULL, this call has no effect.
		\param priority The new priority setting. If this is \c WFMessaging_Priority_Cancel,
			this call is equivalent to \c WFMessaging_Request_clientCancel().
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Request_clientCancel( WFMessaging_RequestRef requestRef );
	/**<
		\brief Cancels a request by calling the \c handlePriorityChange function
		in the worker thread's message handler with the priority of \c WFMessaging_Priority_Cancel.

		Once a request has been canceled, it can not be revived by changing
		its priority.

		\threadsafety Should only be called from the client thread that originated this request.

		\param requestRef The request to update. If \c NULL, this call has no effect.
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Request_clientSetClientInfo( WFMessaging_RequestRef requestRef,
													 const void* clientInfo );
	/**<
		\brief Associates arbitrary developer-defined data with a request, for the client thread's use.

		No other semantics (such as logic to free the pointer) are provided.

		\threadsafety Should only be called from the client thread that originated this request.

		\param requestRef The request to update. If \c NULL, this call has no effect.
		\param clientInfo A pointer to arbitrary developer-defined data.
	*/

//------------------------------------------------------------------------------

WF_API const void* WFMessaging_Request_clientGetClientInfo( WFMessaging_RequestRef requestRef );
	/**<
		\brief Retrieves any developer-defined data that was associated with a request for the client
			thread's use.

		\threadsafety Should only be called from the client thread that originated this request.

		\param requestRef The request. If \c NULL, the call returns \c NULL.

		\return A pointer to developer-defined data, as previously set by a call to
			\c WFMessaging_Request_clientSetClientInfo(), or \c NULL if no such data is associated
			with the request, or if no request is supplied.
	*/

//------------------------------------------------------------------------------

WF_API WFMessaging_RequestRef WFMessaging_Request_workerRetain( WFMessaging_RequestRef requestRef );
	/**<
		\brief Increments the worker reference count for a request.

		Client and worker thread references are counted separately.

		\threadsafety Should only be called from the worker thread that received this request.

		\param requestRef The request to retain. If \c NULL, this call has no effect.

		\return The request object (useful for chaining).
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Request_workerRelease( WFMessaging_RequestRef requestRef );
	/**<
		\brief Decrements the worker reference count for a request. If the
		worker reference count falls to zero, signals the client that the request
		has completed. If the client reference count has also fallen to zero,
		frees the request object.

		\warning Be careful to correctly match client vs worker retains and releases.

		\threadsafety Should only be called from the worker thread that received this request.

		\param requestRef The request to release. If \c NULL, this call has no effect.
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Request_workerReleaseAndClear( WFMessaging_RequestRef* requestRefPtr );
	/**<
		\brief Decrements the worker reference count for a request.
		If the client reference count falls to zero, cancels
		the request. If the worker reference count has also fallen to zero, frees
		the request object.

		\warning Be careful to correctly match client vs worker retains and releases.

		\threadsafety Should only be called from the worker thread that received this request).

		\param requestRefPtr The request to release. If \c NULL, this call has no effect.
			On exit, this pointer is set to \c NULL.
	*/

//------------------------------------------------------------------------------

WF_API bool WFMessaging_Request_workerEnqueueResponse( WFMessaging_RequestRef requestRef,
													   WFMessaging_RequestStatus requestStatus,
													   WFMessaging_Selector statusSelector,
													   WFMessaging_DataRef dataRef );
	/**<
		\brief Sends a request response back to the client thread.

		\threadsafety Should only be called from the worker thread that received this request.

		\param requestRef The request to respond to. If \c NULL, this call has no effect.
		\param requestStatus Reports whether there will be additional responses.
		\param statusSelector The message status, such as "data" or "error".
		\param dataRef Additional developer-defined context data associated with the response,
			or \c NULL.

		\return Returns \c true if the client thread was able to enqueue the response
			for processing; \c false if the client thread has been terminated.
	*/

//------------------------------------------------------------------------------

WF_API void WFMessaging_Request_workerSetWorkerInfo( WFMessaging_RequestRef requestRef,
													 const void* workerInfo );
	/**<
		\brief Associates arbitrary developer-defined data with a request, for the worker thread's use.

		No other semantics (such as logic to free the pointer) are provided.

		\threadsafety Should only be called from the worker thread that received this request.

		\param requestRef The request to update. If \c NULL, this call has no effect.
		\param workerInfo A pointer to arbitrary developer-defined data.
	*/

//------------------------------------------------------------------------------

WF_API const void* WFMessaging_Request_workerGetWorkerInfo( WFMessaging_RequestRef requestRef );
	/**<
		\brief Retrieves any developer-defined data that was associated with a request
		for the worker thread's use.

		\threadsafety Should only be called from the worker thread that received this request.

		\param requestRef The request. If \c NULL, the call returns \c NULL.

		\return A pointer to developer-defined data, as previously set by a call to
			\c WFMessaging_Request_workerSetWorkerInfo(), or \c NULL if no such data is associated
			with the request, or if no request is supplied.
	*/


//==============================================================================

#pragma mark -
#pragma mark external messages

//------------------------------------------------------------------------------

typedef struct WFMessaging_ExternalMessage_t* WFMessaging_ExternalMessageRef;
    /**<
        \brief A mechanism for repeatedly enqueueing an external message to a
		processor.
	*/

//------------------------------------------------------------------------------

WF_API WFMessaging_ExternalMessageRef WFMessaging_ExternalMessage_create(
					WFMessaging_ProcessorRef processorRef,
					WFMessaging_EnqueueOrder enqueueOrder,
					WFMessaging_Selector selector,
					WFMessaging_DataRef dataRef );
    /**<
        \brief Creates an object to repeatedly enqueue an external message.

		\param processorRef The processor targeted by the message.
		\param selector The selector for the message.
		\param dataRef Arbitrary developer-defined data for the message.
		\param enqueueOrder The execution-queue location for the message.
			Enqueue for FIFO execution 	when sequence order matters, and
			for LIFO execution when executing soon matters.

		\return The new external message object.
    */

//------------------------------------------------------------------------------

WF_API WFMessaging_ExternalMessageRef WFMessaging_ExternalMessage_retain(
					WFMessaging_ExternalMessageRef externalMessageRef );
    /**<
        \brief Retains an external message by incrementing its reference count.

		\param externalMessageRef The message object.
		\return The message object (useful for chaining).
    */

//------------------------------------------------------------------------------

WF_API void WFMessaging_ExternalMessage_release(
					WFMessaging_ExternalMessageRef externalMessageRef );
    /**<
        \brief Releases an external message by decrementing its reference count.

		\param externalMessageRef The message object.
    */

//------------------------------------------------------------------------------

WF_API bool WFMessaging_ExternalMessage_enqueue(
					WFMessaging_ExternalMessageRef externalMessageRef );
    /**<
        \brief Enqueues an external message.

		\param externalMessageRef The message object.
		\return Returns \c true on success.
    */


//==============================================================================

#pragma mark -
#pragma mark debugging

//------------------------------------------------------------------------------

WF_API size_t WFMessaging_totalObjects( void );
	/**<
		\brief Reports the total number of objects currently directly allocated
		by the messaging library. (Intended for debugging only.)

		\return A number of object. This number should return to zero when all
		processors have been terminated.
	*/

//------------------------------------------------------------------------------

#endif
