/*******************************************************************************

WFMessagingAtomic.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2011 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

/*!
	\file WFMessagingAtomic.h
	
	\brief \c WFMessagingAtomic provides lock free mechanisms for performing
	various atomic operations such as incrementing and decrementing counters.
*/

#ifndef WFMessaging_Atomic_h
#define WFMessaging_Atomic_h

	 // wichitafoundation/core
#include "WFApiDefs.h"

#include <stdint.h>

#if WIN_ENV || WINRT_ENV
#include <Windows.h>
#endif

//==============================================================================

#if IOS_ENV || MAC_ENV || ANDROID_ENV

typedef volatile int32_t WFMessaging_AtomicCounter;
	/**<
		\brief A \c WFMessaging_AtomicCounter is a counter that is incremented
		and decremented atomically.
	*/

#elif WIN_ENV || WINRT_ENV

typedef volatile LONG WFMessaging_AtomicCounter;
	/**<
		\brief A \c WFMessaging_AtomicCounter is a counter that is incremented
		and decremented atomically.
	*/

#endif

//------------------------------------------------------------------------------

WF_API
WFMessaging_AtomicCounter WFMessaging_AtomicCounter_value(
								WFMessaging_AtomicCounter* counter );
	/**<
		\brief Returns the current value of a \c WFMessaging_AtomicCounter via
		appropriate atomic/synchronized operations. (In other words, it's more
		expensive than just reading the counter.)
	*/

//------------------------------------------------------------------------------

WF_API
WFMessaging_AtomicCounter WFMessaging_AtomicCounter_increment(
								WFMessaging_AtomicCounter* counter );
	/**<
		\brief Increment a \c WFMessaging_AtomicCounter.
		
		\return new value of the counter after increment
	*/
	
//------------------------------------------------------------------------------

WF_API
WFMessaging_AtomicCounter WFMessaging_AtomicCounter_decrement(
								WFMessaging_AtomicCounter* counter );
	/**<
		\brief Decrement a \c WFMessaging_AtomicCounter.

		\return new value of the counter after decrement
	*/
	
//------------------------------------------------------------------------------

WF_API
WFMessaging_AtomicCounter WFMessaging_AtomicCounter_incrementCheckZero(
						WFMessaging_AtomicCounter* counter );
	/**<
		\brief Increment a \c WFMessaging_AtomicCounter.

		The counter should have been at least one to start with. This routine
		will assert if it is not at least that big. This is useful when working
		with reference counts where we should not gain any references if we
		have lost all of the references.

		\return new value of the counter after increment
	*/
						
//------------------------------------------------------------------------------

WF_API
WFMessaging_AtomicCounter WFMessaging_AtomicCounter_decrementCheckZero(
						WFMessaging_AtomicCounter* counter );
	/**<
		\brief Decrement a \c WFMessaging_AtomicCounter returning the new value.

		The counter should have been at least one to start with. This routine
		will assert if it is not at least that big. This is useful when working
		with reference counts where we should not gain any references if we
		have lost all of the references.

		\return new value of the counter after increment
	*/


//==============================================================================

#pragma mark Atomic Accumulators

//------------------------------------------------------------------------------

#if IOS_ENV || MAC_ENV || ANDROID_ENV

typedef volatile int64_t WFMessaging_AtomicAccumulator;
	/**<
		\brief A \c WFMessaging_AtomicAccumulator is a value that is adjusted
		atomically generally to track space usage.
	*/

#elif WIN_ENV || WINRT_ENV

typedef volatile LONGLONG WFMessaging_AtomicAccumulator;
	/**<
		\brief A \c WFMessaging_AtomicAccumulator is a value that is adjusted
		atomically generally to track space usage.
	*/

#endif

//------------------------------------------------------------------------------

WF_API
WFMessaging_AtomicAccumulator WFMessaging_AtomicAccumulator_value(
								WFMessaging_AtomicAccumulator* counter );
	/**<
		\brief Returns the current value of a \c WFMessaging_AtomicAccumulator via
		appropriate atomic/synchronized operations. (In other words, it's more
		expensive than just reading the accumulator.)
	*/

//------------------------------------------------------------------------------

WF_API
WFMessaging_AtomicAccumulator WFMessaging_AtomicAccumulator_adjust(
								WFMessaging_AtomicAccumulator* accumulator,
								WFMessaging_AtomicAccumulator delta );
	/**<
		\brief Adjust a \c WFMessaging_AtomicAccumulator.
		
		\return new value of the accumulator after adjustment
	*/
	
//==============================================================================

#pragma mark Atomic IDs

//------------------------------------------------------------------------------

typedef int64_t WFMessaging_AtomicId;
	/**<
		\brief \c WFMessaging_AtomicID is the type for an app-wide incrementing ID.
		
		Values are guaranteed to be unique during the lifetime of the application.
		
		Note: No volatile here since id's should more or less be things unto
		themselves.
	*/

//------------------------------------------------------------------------------

WF_API
WFMessaging_AtomicId WFMessaging_AtomicId_generate( void );
	/**<
		\brief Increment the ID generator and return the next ID.
		
		This value can safely be used throughout the lifetime of this application
		session as a unique ID. No call to this function on any thread will
		receive this value again.
	*/


//==============================================================================

#pragma mark Atomic Pointers

//------------------------------------------------------------------------------

typedef void * volatile WFMessaging_AtomicPointer;

//------------------------------------------------------------------------------

WF_API
WFMessaging_AtomicPointer WFMessaging_AtomicPointer_getCachedPointer(
								WFMessaging_AtomicPointer* storage,
								WFMessaging_AtomicPointer (*calc)( void* ),
								void* calcData,
								void (*free)( void*, WFMessaging_AtomicPointer ) );
	/**<
		\brief Get or calculate a cached pointer value. If the cache has
		already been successfully filled, this just returns the cache contents.
		If not, it calculates the contents and fills the cache. This does not
		protect against race-conditions calculating the cache contents more than
		once but in that event, the free routine will be invoked on the superfluous
		result.
	*/


//==============================================================================

#endif
