/*******************************************************************************

WFMessagingData.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2011 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

/*!
	\file WFMessagingData.h
	
	\brief \c WFMessagingData provides a standard mechanism for handing around
	blocks of data used as parameters to the messaging APIs.
*/

#ifndef WFMessaging_Data_h
#define WFMessaging_Data_h

	// wichitafoundation/core/threading
#include "WFMessagingSelectors.h"

	// C standard library
#include <stddef.h>


//==============================================================================

typedef struct WFMessaging_Data_t* WFMessaging_DataRef;
	/**<
		\brief A reference to a data object – essentially a container for an
		uninterpreted blob of bytes that can be freed.
		
		A data ref may have a format type if \c WFMessaging_Data_createWithFormat()
		is called to create the data object with a non-\c NULL \c format specifier.
	*/

//------------------------------------------------------------------------------

typedef void (*WFMessaging_Data_FreeProc)( const void* bytes,
										   size_t length,
										   const void* ctx );
	/**<
		\brief This function is passed to \c WFMessaging_Data_create() and
		called when the corresponding data object is freed. It should release any
		resources associated with this data block.
		
		\param bytes pointer to data as presented to \c WFMessaging_Data_create()
		\param length length of data as presented to \c WFMessaging_Data_create()
		\param ctx context pointer as presented to \c WFMessaging_Data_create()
	*/

//------------------------------------------------------------------------------

WF_API
WFMessaging_DataRef WFMessaging_Data_create( WFMessaging_Data_FreeProc freeProc,
											 const void* bytes,
											 size_t length,
											 const void* ctx );
	/**<
		\brief Create a data object.

		\param freeProc function that will be called when this data object is
			released. Must be non-\c NULL.
		\param bytes pointer to raw data (may be \c NULL, in which case the
			remaining parameters don't matter)
		\param length number of bytes at \c bytes
		\param ctx context pointer to be handed to \c freeProc at release time.
			Not otherwise interpreted by this code.

		\return reference to data object or \c NULL if \c bytes is \c NULL.
	*/

//------------------------------------------------------------------------------

WF_API
WFMessaging_DataRef WFMessaging_Data_createWithFormat(
                        WFMessaging_Selector format,
                        WFMessaging_Data_FreeProc freeProc,
						const void* bytes,
						size_t length,
						const void* ctx );
	/**<
		\brief Create a data object with a format specifier.
		
		The format specifier can be queried via \c WFMessaging_Data_format()
		if the caller needs to verify that a specific encoding was used.

		\param freeProc function that will be called when this data object is
			released. Must be non-\c NULL.
		\param format format specifier to be returned via \c WFMessaging_Data_format().
			May be \c NULL.
		\param bytes pointer to raw data (may be \c NULL, in which case the
			remaining parameters don't matter)
		\param length number of bytes at \c bytes
		\param ctx context pointer to be handed to \c freeProc at release time.
			Not otherwise interpreted by this code.

		\return reference to data object or \c NULL if \c bytes is \c NULL.
	*/

//------------------------------------------------------------------------------

WF_API
WFMessaging_DataRef WFMessaging_Data_createByTakingOwnership( const void* bytes,
															  size_t length );
	/**<
		\brief Create a data object and assume ownership of the data block that
		is passed in.

		When this data reference loses its last reference, the memory block will
		be destroyed by calling <code>free( bytes )</code>.
		
		\param bytes must point to a block of data allocated using \c malloc()
			or \c NULL.
		\param length number of bytes in this block
		
		\return reference to data object or \c NULL if \c bytes is \c NULL.
	*/

//------------------------------------------------------------------------------

WF_API
WFMessaging_DataRef WFMessaging_Data_createByCopying( const void* bytes,
													  size_t length );
	/**<
		\brief Copy data referenced by \c bytes into a new block and create
		a data object to track it.
		
		\param bytes data block (data will be copied during this call). The data
			object does \b not assume ownership of the memory block pointed to
			by \c bytes. Once this call completes, you may release or alter the
			data at \c bytes at any time.
		\param length number of bytes in this block
		
		\return reference to data object or \c NULL if \c bytes is \c NULL.
	*/

//------------------------------------------------------------------------------

WF_API
WFMessaging_DataRef WFMessaging_Data_createByCopyingString( const char* text );
	/**<
		\brief Copy the string as a data object.
		
		\param text must point to a <code>NULL</code>-terminated string
			or \c NULL. The data object does \b not assume ownership of the string
			pointed to by \c text. Once this call completes, you may release or
			alter the string at \c text at any time.
		  
		\return reference to data object or \c NULL if \c text is \c NULL.
	*/

//------------------------------------------------------------------------------

WF_API
WFMessaging_DataRef WFMessaging_Data_retain( WFMessaging_DataRef dataRef );
	/**<
		\brief Increment the reference count for \c dataRef.
		
		\param dataRef data object to retain. May be \c NULL, in which case
			this function is a no-op.
		
		\return \c dataRef (useful for chaining)
	*/

//------------------------------------------------------------------------------

WF_API
void WFMessaging_Data_release( WFMessaging_DataRef dataRef );
	/**<
		\brief Decrement the reference count for \c dataRef and free it if
		the counter reaches zero.
		
		\param dataRef data object to release. May be \c NULL, in which case
			this function is a no-op.
	*/

//------------------------------------------------------------------------------

WF_API
void WFMessaging_Data_releaseAndClear( WFMessaging_DataRef* dataRefPtr );
	/**<
		\brief Decrement the reference count for the data object at \c *dataRefPtr
		and free it if the counter reaches zero.
		
		\param dataRefPtr pointer to variable that may refer to a data object to
			be released. Either \c dataRef or \c *dataRef may be \c NULL, in which
			case this function is a no-op. On exit, \c *dataRef will be set to
			\c NULL (assuming, of course, that \c dataRef itself was non-\c NULL).
	*/

//------------------------------------------------------------------------------

WFMessaging_Selector WFMessaging_Data_format( WFMessaging_DataRef dataRef );
	/**<
		\brief Returns the format specifier (if any) provided when this
		data object was created.
		
		\param dataRef data object to query. May be \c NULL.
	
		\return format specifier that was specified when this data object was
			created (only if created via \c WFMessaging_Data_createWithFormat())
			or \c NULL otherwise.
	*/

//------------------------------------------------------------------------------

WF_API
const void* WFMessaging_Data_context( WFMessaging_DataRef dataRef );
	/**<
		\brief Returns the context value (if any) provided when this
		data object was created.
		
		\param dataRef data object to query. May be \c NULL.
	
		\return context value that was specified when this data object was
			created.
	*/

//------------------------------------------------------------------------------
	
WF_API
const void* WFMessaging_Data_bytes( WFMessaging_DataRef dataRef );
	/**<
		\brief Returns the raw pointer data for this data object.

		\param dataRef data object to query. May be \c NULL.
	
		\return pointer to raw data for this data object or \c NULL if
			\c dataRef was \c NULL
	*/

//------------------------------------------------------------------------------

WF_API
size_t WFMessaging_Data_length( WFMessaging_DataRef dataRef );
	/**<
		\brief Returns the number of bytes held by this data object.

		\param dataRef data object to query. May be \c NULL.
	
		\return pointer to raw data for this data object or zero if
			\c dataRef was \c NULL.
	*/


//==============================================================================

WF_API
size_t WFMessaging_Data_totalSpace( void );
	/**<
		\brief Returns the total amount of space in managed by WFMessaging_Data
		objects.
	*/
	

//==============================================================================

#endif
