/*******************************************************************************

WFMessagingImpDispatch.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2011 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

/*!
	\file WFMessagingImpDispatch.h
	
	\brief \c WFMessagingImpDispatch provides a wrapper around platform threading
	services.
*/

#ifndef WFMessagingImpDispatch_h
#define WFMessagingImpDispatch_h

#include "WFMessagingThreadPriorities.h"

#include <stddef.h>


//==============================================================================

typedef struct WFMessagingImpDispatch_Mutex_t* WFMessagingImpDispatch_MutexRef;
	/**< \brief A basic mutex. */

//------------------------------------------------------------------------------

WFMessagingImpDispatch_MutexRef WFMessagingImpDispatch_Mutex_create(
							const char* optDescription );
	/**<
		\brief Create a mutex.
		
		\param optDescription a string label to attach to the mutex (for debugging
			purposes only). This parameter is optional and may be \c NULL.
		
		\return mutex reference
	*/

//------------------------------------------------------------------------------

WFMessagingImpDispatch_MutexRef WFMessagingImpDispatch_Mutex_retain(
							WFMessagingImpDispatch_MutexRef mutexRef );
	/**<
		\brief Add a reference to an existing mutex object.
		
		\param mutexRef a mutex created via \c WFMessagingImpDispatch_Mutex_create().
			May be \c NULL, in which case this function is a no-op.
		
		\return mutexRef
	*/

//------------------------------------------------------------------------------

void WFMessagingImpDispatch_Mutex_release( WFMessagingImpDispatch_MutexRef mutexRef );
	/**<
		\brief Release a reference to an existing mutex object.
		
		If the mutex's reference count drops to zero, then the mutex is destroyed.
		
		\param mutexRef a mutex created via \c WFMessagingImpDispatch_Mutex_create().
			May be \c NULL, in which case this function is a no-op.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpDispatch_Mutex_releaseAndClear(
							WFMessagingImpDispatch_MutexRef* mutexRefPtr );
	/**<
		\brief Decrement the reference count for the mutex at \c *mutexRefPtr
		and free it if the counter reaches zero.
		
		\param mutexRefPtr pointer to variable that may refer to a mutex object to
			be released. Either \c mutexRefPtr or \c *mutexRefPtr may be \c NULL,
			in which case this function is a no-op. On exit, \c *mutexRefPtr will
			be set to \c NULL (assuming, of course, that \c mutexRefPtr itself was
			non-\c NULL).
	*/

//------------------------------------------------------------------------------

void WFMessagingImpDispatch_Mutex_do( WFMessagingImpDispatch_MutexRef mutexRef,
									  void* pb,
									  void (*func)( void* ) );
	/**<
		\brief Perform the function when this mutex becomes available.
		
		This call will block the calling thread until the mutex becomes
		available.
		
		\param mutexRef a mutex created via \c WFMessagingImpDispatch_Mutex_create().
			Must not be \c NULL.
		\param pb arbitrary data pointer to be passed to \c func. Not interpreted
			by this code. May be \c NULL.
		\param func function to be called when mutex is available. Must not be
			\c NULL.
	*/


//==============================================================================

typedef struct WFMessagingImpDispatch_Queue_t* WFMessagingImpDispatch_QueueRef;
	/**< \brief A task queue (thread) that processes tasks in FIFO order. */

//------------------------------------------------------------------------------

WFMessagingImpDispatch_QueueRef WFMessagingImpDispatch_Queue_create(
							const char* optDescription );
	/**<
		\brief Create a task queue.
		
		\param optDescription a string label to attach to the queue (for debugging
			purposes only). This parameter is optional and may be \c NULL.
		
		\return task queue reference
	*/

//------------------------------------------------------------------------------

WFMessagingImpDispatch_QueueRef WFMessagingImpDispatch_Queue_retain(
							WFMessagingImpDispatch_QueueRef queueRef );
	/**<
		\brief Add a reference to an existing task queue object.
		
		\param queueRef a task queue created via \c WFMessagingImpDispatch_Queue_create().
			May be \c NULL, in which case this function is a no-op.
		
		\return queueRef
	*/

//------------------------------------------------------------------------------

void WFMessagingImpDispatch_Queue_release( WFMessagingImpDispatch_QueueRef queueRef );
	/**<
		\brief Release a reference to an existing task queue object.
		
		If the task queue's reference count drops to zero, then the task queue
		is destroyed.
		
		\param queueRef a task queue created via \c WFMessagingImpDispatch_Queue_create().
			May be \c NULL, in which case this function is a no-op.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpDispatch_Queue_releaseAndClear(
							WFMessagingImpDispatch_QueueRef* queueRefPtr );
	/**<
		\brief Decrement the reference count for the task queue at \c *queueRefPtr
		and free it if the counter reaches zero.
		
		\param queueRefPtr pointer to variable that may refer to a task queue object
			to be released. Either \c queueRefPtr or \c *queueRefPtr may be \c NULL,
			in which case this function is a no-op. On exit, \c *queueRefPtr will
			be set to \c NULL (assuming, of course, that \c queueRefPtr itself was
			non-\c NULL).
	*/

//------------------------------------------------------------------------------

void WFMessagingImpDispatch_Queue_syncDo( WFMessagingImpDispatch_QueueRef queueRef,
										  void* pb,
										  void (*func)( void* ) );
	/**<
		\brief Perform the function in the task queue. Block the calling thread
		until the function completes.
		
		\param queueRef a task queue created via \c WFMessagingImpDispatch_Queue_create().
			Must not be \c NULL.
		\param pb arbitrary data pointer to be passed to \c func. Not interpreted
			by this code. May be \c NULL.
		\param func function to be called on the task queue. Must not be \c NULL.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpDispatch_Queue_asyncDo( WFMessagingImpDispatch_QueueRef queueRef,
										   void* pb,
										   void (*func)( void* ) );
	/**<
		\brief Perform the function in the task queue. Does not block the calling
		thread.
		
		This API does not provide any notification when the task has been completed.
		
		\param queueRef a task queue created via \c WFMessagingImpDispatch_Queue_create().
			Must not be \c NULL.
		\param pb arbitrary data pointer to be passed to \c func. Not interpreted
			by this code. May be \c NULL.
		\param func function to be called on the task queue. Must not be \c NULL.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpDispatch_Queue_afterDelayDo( WFMessagingImpDispatch_QueueRef queueRef,
											    double delay,
												void* pb,
											    void (*func)( void* ) );
	/**<
		\brief Perform the function after the specified delay. Does not block
		the calling thread.
		
		This API does not provide any notification when the task has been completed.

		There is no mechanism to cancel execution of this function once it has
		been scheduled.
		
		\threadsafety May be called from any thread.

		\param queueRef a task queue created via \c WFMessagingImpDispatch_Queue_create().
			Must not be \c NULL.
		\param delay how long to wait (in seconds) before scheduling execution of
			this function. Must not a positive (non-zero) number.
		\param pb arbitrary data pointer to be passed to \c func. Not interpreted
			by this code. May be \c NULL.
		\param func function to be called on the task queue. Must not be \c NULL.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpDispatch_Queue_setPriority( WFMessagingImpDispatch_QueueRef queueRef,
											   WFMessaging_ThreadPriority priority );
	/**<
		\brief Change the priority for this task queue.
		
		\threadsafety This function may be called from any thread.
		
		\param queueRef a task queue created via \c WFMessagingImpDispatch_Queue_create().
			Must not be \c NULL.
		\param priority the new thread priority. Must be one of the constants
			defined by \c WFMessaging_ThreadPriority.
	*/
		

//==============================================================================

#if COCOA_ENV
#pragma mark -
#pragma mark Debugging information
#endif

//------------------------------------------------------------------------------

size_t WFMessagingImpDispatch_totalDispatchObjects( void );
	/**<
		\brief Returns the total number of outstanding retain requests for
		objects created via \c WFMessagingImpDispatch_Mutex_create() or
		\c WFMessagingImpDispatch_Queue_create.
		
		This function is intended for debugging use.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpDispatch_decrementObjectCountForPersistentGlobalObject( void );
	/**<
		\brief Provide a debugging hint that an object created via the dispatch
		core code should live for the entire lifetime of the application.
		
		This call is intended for internal use only.
	*/


//==============================================================================

#endif
