/*******************************************************************************

WFMessagingImpUtils.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2011 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

/*!
	\file WFMessagingImpUtils.h
	
	\brief \c WFMessagingImpUtils contains internal utilities for WFMessaging.
*/

#ifndef WFMessagingImpUtils_h
#define WFMessagingImpUtils_h

#include <stddef.h>
#include "WFMessagingAtomic.h"


//==============================================================================

void* WFMessagingImpUtils_allocateObject(
								size_t size,
								WFMessaging_AtomicCounter* counter );
	/**<
		\brief Private utility for <code>malloc</code>'ing an object of \c size
		bytes. This allocation is tracked in a count of the total number of
		allocated objects.
		
		This object should be freed using \c WFMessagingImpUtils_freeObject()
		instead of \c free.
	*/
	
//------------------------------------------------------------------------------

#define WFMessagingImpUtils_allocateObjectOfType( T, counter ) \
	(T *) WFMessagingImpUtils_allocateObject( sizeof( T ), counter )
	/**<
		\brief Private utility macro for <code>malloc</code>'ing an object of a
		particular type. This in turn calls \c WFMessagingImpUtils_allocateObject().
		
		This object should be freed using \c WFMessagingImpUtils_freeObject()
		instead of \c free.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpUtils_freeObject(
						void* obj, WFMessaging_AtomicCounter* counter );
	/**<
		\brief Free an allocated object. Does nothing if \c obj is \c NULL.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpUtils_leakPersistentObject(
						void* obj, WFMessaging_AtomicCounter* counter );
	/**<
		\brief Remove an allocated object from the count on the basis that it
		will persist indefinitely. (It's a bad thing to then free it later, so
		use this with care. If \c obj is \c NULL, then this routine does nothing.
	*/

//------------------------------------------------------------------------------

size_t WFMessagingImpUtils_totalObjects( void );
	/**<
		\brief Returns the number of objects allocated via
		\c WFMessagingImpUtils_allocateObject() that have not been released via
		\c WFMessagingImpUtils_freeObject() or intentionally "leaked" via
		\c WFMessagingImpUtils_leakPersistentObject().
	*/


//==============================================================================

typedef struct WFMessagingImpUtils_Registry_t* WFMessagingImpUtils_RegistryRef;
	/**<
		\brief \c WFMessagingImpUtils_RegistryRef is a wrapper for a
		<code>void*</code> to <code>void*</code> registry. It does not provide
		any mutex protection.
	*/

//------------------------------------------------------------------------------

WFMessagingImpUtils_RegistryRef WFMessagingImpUtils_Registry_create( void );
	/**<
		\brief Create a registry.
		
		\sa WFMessagingImpUtils_RegistryRef
	*/

//------------------------------------------------------------------------------

WFMessagingImpUtils_RegistryRef WFMessagingImpUtils_Registry_retain(
				WFMessagingImpUtils_RegistryRef registryRef );
	/**<
		\brief Add a reference to a registry.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpUtils_Registry_release(
				WFMessagingImpUtils_RegistryRef registryRef );
	/**<
		\brief Remove a reference to a registry.
		
		If the reference count falls to zero, the registry is destroyed.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpUtils_Registry_releaseAndClear(
				WFMessagingImpUtils_RegistryRef* registryRefPtr );
	/**<
		\brief Remove a reference to a registry and clear the reference.
		
		If the reference count falls to zero, the registry is destroyed.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpUtils_Register_leakPersistentRegistry(
		WFMessagingImpUtils_RegistryRef registryRef );
	/**<
		\brief Remove a registry from the object tracking so that it isn't
		registered as an unintended leak.
	*/

//------------------------------------------------------------------------------

const void* WFMessagingImpUtils_Registry_getEntry(
				WFMessagingImpUtils_RegistryRef registryRef,
				const void* key );
	/**<
		\brief Get a registry entry or <code>NULL</code> if nothing is present.
		
		A registry entry will exist only if previously created via
		\c WFMessagingImpUtils_Registry_setEntry().
	*/

//------------------------------------------------------------------------------

void WFMessagingImpUtils_Registry_setEntry(
				WFMessagingImpUtils_RegistryRef registryRef,
				const void* key,
				const void* value );
	/**<
		\brief Set a registry entry.
	*/

//------------------------------------------------------------------------------

void WFMessagingImpUtils_forEachEntryDo( WFMessagingImpUtils_RegistryRef registryRef,
										 void (*func)( const void* key,
													   const void* entry,
													   void* pb ),
										 void* pb );
	/**\
		\brief Call \c func for each entry in the registry.
	*/


//==============================================================================

#endif
