/*******************************************************************************

WFMessagingOnce.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2011 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

/*!
	\file WFMessagingOnce.h
	
	\brief \c WFMessagingOnce provides standard unique execution services and services
	for working with uniquely allocated mutexes.
*/

#ifndef WFMessagingOnce_h
#define WFMessagingOnce_h

#include "WFApiDefs.h"

//==============================================================================

#pragma mark -
#pragma mark Duplicate task avoidance

//------------------------------------------------------------------------------

/**<
 \brief A semaphore used by \c WFMessagingImpDispatch_Once_do() to ensure
 that a task is only performed once.
 */

#if IOS_ENV || MAC_ENV

typedef long WFMessagingOnce_imp_t;
    // NOTE: Must match dispatch_once_t!

#define WFMessagingOnce_imp_initValue 0

#elif WIN_ENV || WINRT_ENV

#include <windows.h>

typedef INIT_ONCE WFMessagingOnce_imp_t;

#define WFMessagingOnce_imp_initValue INIT_ONCE_STATIC_INIT

#elif ANDROID_ENV

#include <pthread.h>

typedef struct {
	pthread_mutex_t		singleShot;
	pthread_mutex_t		exclusion;
} WFMessagingOnce_imp_t;
#define WFMessagingOnce_imp_initValue	PTHREAD_MUTEX_INITIALIZER

#endif

/**<
 \brief Allocate and initialize a static once value.
*/

#define WFMessagingOnce_Decl( name ) \
    static WFMessagingOnce_imp_t name = WFMessagingOnce_imp_initValue;

//------------------------------------------------------------------------------

WF_API void WFMessagingOnce_do( WFMessagingOnce_imp_t* once,
				void* pb,
				void (*func)( void* ) );
	/**<
		\brief Perform a function, but only once per application session.
		
		If this function is called multiple times using the same \c once
		semaphore, only one call to \c func will occur, even if the multiple
		attempts originate simulataneously from multiple threads. The second
		and succeeding call to this function with the same \c once semaphore
		will be no-ops.
		
		The calling thread is blocked until \c func succeeds.
		
		\param once semaphore variable
		\param pb arbitrary data pointer to be passed to \c func. Not interpreted
			by this code. May be \c NULL.
		\param func function to be called on the task queue. Must not be \c NULL.
	*/


//==============================================================================

WF_API void WFMessagingOnce_globalMutexDo_imp(
				WFMessagingOnce_imp_t* once, // Mutex allocation control
				const char* mutexName,
				void** mutexStorage,
				void* pb,
				void (*func)( void* ) );
	/**<
		\brief Perform a function subject to a global singly-allocated mutex.
	*/
	
#define WFMessagingOnce_globalMutexDo( mutexName, pb, func ) \
{ \
	WFMessagingOnce_Decl( mutexName ## Once ); static void* mutexName ## Mutex = NULL; \
	WFMessagingOnce_globalMutexDo_imp( &mutexName ## Once, "com.adobe.wichita." #mutexName, &mutexName ## Mutex, pb, func ); \
}

//==============================================================================

#endif
