/*******************************************************************************

WFMessagingSelectors.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2011 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

/*!
	\file WFMessagingSelectors.h
	
	\brief \c WFMessagingSelectors provides a thread-safe mechanism for interning
	strings. These selectors are used in the messaging library.
*/

#ifndef WFMessaging_Selectors_h
#define WFMessaging_Selectors_h

#include "WFMessagingAtomic.h"
#include "WFApiDefs.h"


//==============================================================================

typedef struct WFMessaging_Selector_t* WFMessaging_Selector;
	/**<
		\brief A \c WFMessaging_Selector is an interned string. */

//------------------------------------------------------------------------------

WF_API
WFMessaging_Selector WFMessaging_Selector_register( const char* text );
	/**<
		\brief Convert a plain-text string into a \c WFMessaging_Selector.
	
		If you register the same string (regardless of pointer equality),
		the \c WFMessaging_Selector value that is returned will match.
		
		If \c text is \c NULL, then the returned selector value will also be \c NULL.
	*/

//------------------------------------------------------------------------------

WF_API
WFMessaging_Selector WFMessaging_Selector_registerConst(
							WFMessaging_AtomicPointer* storage,
							const char* constText );
	/**<
		\brief Convert a constant plain-text string into a \c WFMessaging_Selector.
	
		This routine will cache the value in the provided storage so if the storage
		is allocated statically and initialized to NULL (as it should be), we will
		only perform the initialization a limited number of times (ideally 1 but
		threading issues could cause us to try registering more than once).
		
		If \c constText is \c NULL, then the returned selector value will also be \c NULL.
	*/

//------------------------------------------------------------------------------

#define WFMessaging_Selector_define_label_string( label, text ) \
static WFMessaging_Selector selector_ ## label ( void ) { \
	static WFMessaging_AtomicPointer storage = NULL; \
	return WFMessaging_Selector_registerConst( &storage, text ); \
}

#define WFMessaging_Selector_define( x ) WFMessaging_Selector_define_label_string( x, #x )

//------------------------------------------------------------------------------

WF_API
const char* WFMessaging_Selector_toText( WFMessaging_Selector selector );
	/**<
		\brief Convert a \c WFMessaging_Selector back to its original string.

		If \c selector is \c NULL, then the returned text value will also be \c NULL.
	*/

//------------------------------------------------------------------------------

#endif
