/*******************************************************************************

WFModels.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2011 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

#ifndef WFModels_h
#define WFModels_h

#include "WFApiDefs.h"

#if COCOA_ENV || ANDROID_ENV
#include <stdbool.h>
#endif

struct lua_State;


//==============================================================================

typedef struct WFModels_Factory WFModels_Factory;
typedef struct WFModels_Model WFModels_Model;


//==============================================================================

// Abstraction over client data. This is data handed from the client to the
// model. It is described based on a pointer to a pointer to a method table.

typedef struct WFModels_ClientDataMethods const * WFModels_ClientData;

//------------------------------------------------------------------------------

// Abstraction over server data as encoded by the protocol and passed to the
// client side.

typedef struct WFModels_ServerDataMethods const * WFModels_ServerData;


//==============================================================================

// Abstraction over the server. This handles all of the server side processing
// for factories and models.

typedef struct WFModels_ServerMethods const * WFModels_Server;

//------------------------------------------------------------------------------

// Abstraction over the client. This handles the client side processing for
// models in general. The protocol handles the actual data delivery.

typedef struct WFModels_ClientMethods const * WFModels_Client;

//------------------------------------------------------------------------------

// Abstraction over the protocol for transferring data from server to client.
// This object both encodes data and errors (generally on the server side) and
// receives those encoded values on the client side.

typedef struct WFModels_ProtocolMethods const * WFModels_Protocol;


//==============================================================================

// Abstraction over mutexes. These are created and destroyed on the client
// side but used by both the client and the server.

//------------------------------------------------------------------------------

typedef struct WFModels_MutexMethods const * WFModels_Mutex;


//==============================================================================

// If we name the methods field _methods, then we can get from a concrete
// type to the abstract type using the following macro:

#define WFModels_asAbstract( v ) ( &( (v)->_methods ) )

//==============================================================================

// We declare actions that run in a variety of contexts. In particular note that
// the client and server actions also expect the scheduler to provide the
// client or server respectively to the action when it is invoked. We do not
// expect this with mutexes.

//------------------------------------------------------------------------------

typedef void (*WFModels_MutexAction)( void * actionData );

typedef void (*WFModels_ClientAction)( WFModels_Client * client, void * actionData );

typedef void (*WFModels_ServerAction)( WFModels_Server * server, void * actionData );

typedef void (*WFModels_ActionDataCleanup)( void * actionData );

//==============================================================================

// Method declarations for above abstract types.

//------------------------------------------------------------------------------

// Client data can be destroyed or decoded. The decode logic should be able
// to assume that we will decode at most once.

typedef struct WFModels_ClientDataMethods {
	void (*destroyClientData)( WFModels_ClientData * data );
	void (*decodeClientDataToLua)( WFModels_ClientData *, struct lua_State * );
} WFModels_ClientDataMethods;

//------------------------------------------------------------------------------

// Server data can only be destroyed (as far as this code is concerned). Specific
// protocols which construct the server data will presumably want to cast the
// delivered data and make use of the underlying value.

typedef struct WFModels_ServerDataMethods {
	void (*destroyServerData)( WFModels_ServerData * data );
} WFModels_ServerDataMethods;

//------------------------------------------------------------------------------

// We can destroy mutexes and we can work with them while locked. We will
// destroy the mutex from the client thread just as we created it from the
// client thread.

typedef struct WFModels_MutexMethods {
		void (*destroyMutex_client)( WFModels_Mutex * mutex );
		void (*mutexDo)( WFModels_Mutex * mutex, WFModels_MutexAction action, void * actionData );
} WFModels_MutexMethods;


//==============================================================================

// Utility routines for invoking methods. Most methods are only expected to
// be invoked internally and hence don't get this treatment.

//------------------------------------------------------------------------------

void WFModels_ClientData_destroy( WFModels_ClientData * clientData );
void WFModels_ClientData_decodeToLua( WFModels_ClientData * clientData, struct lua_State * L );

//------------------------------------------------------------------------------

void WFModels_ServerData_destroy( WFModels_ServerData * serverData );


//==============================================================================

// Server methods.

typedef struct WFModels_ServerMethods {

	// Schedule an action to run on the server thread. If scheduleEarly is
	// true, the server is encouraged but not required to schedule the
	// action early. It basically indicates that the action isn't sequence
	// dependent on other actions and it is deemed likely to be helpful
	// to schedule it to execute earlier rather than later.

	void (*scheduleServerAction)(
				WFModels_Server * server,
				bool scheduleEarly,
				WFModels_ServerAction action,
				void * actionData,
				WFModels_ActionDataCleanup cleanup );
				
	// Server Thread: Destroy the server (as far as this factory usage is concerned).
	
	void (*serverDestroyServer)( WFModels_Server * server );
	
	// Server thread: Open a model. The server must take ownership of the target
	// model if only to then post an error and close it.
	
	void (*serverOpenModel)(
				WFModels_Server * server,
				WFModels_Model * targetModel,
				WFModels_Model * refModel,
				const char * selector,
				WFModels_ClientData * params );
				
	// Server thread: Receive notification that the client closed a model.
	// Generally, the server will then close its side of the model. This message
	// will not be delivered if the server has already closed the model.
	
	void (*serverClientDidCloseModel)(
				WFModels_Server * server,
				WFModels_Model * model );
				
	// Server thread: Receive an event from the client.
	
	void (*serverReceiveEvent)(
				WFModels_Server * server,
				WFModels_Model * model,
				const char * selector,
				WFModels_ClientData * params );
				
} WFModels_ServerMethods;


//==============================================================================

// Client methods.

typedef struct WFModels_ClientMethods {

	// Schedule an action to run on the client thread.

	void (*scheduleClientAction)(
				void * clientHook,
				WFModels_Client * client,
				WFModels_ClientAction action,
				void * actionData,
				WFModels_ActionDataCleanup cleanup );
				
	// Client thread: Destroy the client (at least as far as this factory and
	// its models are concerned).
	
	void (*clientDestroyClient)( WFModels_Client * client );
	
	// Client thread: Create a mutex.
	
	WFModels_Mutex * (*clientCreateMutex)( WFModels_Client * client );
	
	// Note that most of the actual client behavior is covered in the protocol
	// which is kept separate to allow it to be customized for different types
	// of data.

} WFModels_ClientMethods;


//==============================================================================

// Protocol methods.

typedef struct WFModels_ProtocolMethods {

	// Destroy the protocol. Can be called from any thread. If you just have
	// a global protocol, do nothing in this routine (or use
	// WFModels_Protocol_destroyGlobalProtocolNOP).

	void (*destroyProtocol)( WFModels_Protocol * protocol );
	
	// Server thread: Encode data from the server.
	
	WFModels_ServerData * (*encodeDataFromLua_server)(
				WFModels_Protocol * protocol,
				struct lua_State* L,
				int base );

	// Client thread: Receive data from the server. The code can either take
	// possession of the server data or leave it to be destroyed by the
	// calling code. The clientHook is specified when creating the model.
	
	void (*deliverDataToClient)(
				WFModels_Protocol * protocol,
				WFModels_Client * client,
				WFModels_Model * model,
				void * clientHook,
				WFModels_ServerData ** serverDataRef );
				
	// Client thread: Receive an error from the server. The code can either take
	// possession of the server data or leave it to be destroyed by the
	// calling code. The clientHook is specified when creating the model.
	
	void (*deliverErrorToClient)(
				WFModels_Protocol * protocol,
				WFModels_Client * client,
				WFModels_Model * model,
				void * clientHook,
				const char * message );
				
	// Client thread: Receive notification that the server did close the model.
	// The code should generally respond by closing the model on the client
	// side.
	
	void (*deliverDoneToClient)(
				WFModels_Protocol * protocol,
				WFModels_Client * client,
				WFModels_Model * model,
				void * clientHook );

} WFModels_ProtocolMethods;

//------------------------------------------------------------------------------

// A no-op destroy routine for prootocols that you expect to use repeatedly
// without reallocation.

void WFModels_Protocol_destroyGlobalProtocolNOP( WFModels_Protocol * protocol );


//==============================================================================

// We provide simple implementations for data that is encoded from a string.

//------------------------------------------------------------------------------

WF_API WFModels_ClientData * WFModels_createStringClientData( const char * text );

//------------------------------------------------------------------------------

WF_API WFModels_ServerData * WFModels_createStringServerData( const char * text );

//------------------------------------------------------------------------------

WF_API const char * WFModels_StringClientData_getText(
							WFModels_ClientData * clientData );

//------------------------------------------------------------------------------

WF_API const char * WFModels_StringServerData_getText(
							WFModels_ServerData * serverData );


//==============================================================================

// Open a model factory. Both the client and server object are owned by and will
// be destroyed when the factory and any models created through it finishes
// closing.

WF_API WFModels_Factory* WFModels_Factory_clientOpen(
	WFModels_Client * client,
	WFModels_Server * server );
	
//------------------------------------------------------------------------------

// Close a model factory. Note that if it has any models left open these are
// not automatically closed.

void WFModels_Factory_clientClose( WFModels_Factory* factory );
	

//==============================================================================

// Open a model from the factory.

WF_API WFModels_Model * WFModels_Factory_clientOpenModel(
		WFModels_Factory * factory,
		const char * selector,
		WFModels_ClientData * params,
		WFModels_Protocol * protocol,
		void * clientHook );
		
//------------------------------------------------------------------------------

// Open a model from another model (useful for passing on context).

WF_API WFModels_Model * WFModels_Model_clientOpenModel(
		WFModels_Model * model,
		const char * selector,
		WFModels_ClientData * params,
		WFModels_Protocol * protocol,
		void * clientHook );
		
//------------------------------------------------------------------------------

// Close a model from the client side.

WF_API void WFModels_Model_clientClose( WFModels_Model * model );

//------------------------------------------------------------------------------

// Close a model from the server side. This is much more rare other than in
// response to the client being closed.

WF_API void WFModels_Model_serverClose( WFModels_Model * model );

//------------------------------------------------------------------------------

// Post data to a model. ( server -> client )

WF_API void WFModels_Model_serverPostDataFromLua(
				WFModels_Model * model,
				struct lua_State * L,
				int base );

//------------------------------------------------------------------------------

// Post an error to a model. ( server -> client )

WF_API void WFModels_Model_serverPostError( WFModels_Model * model, const char * message );

//------------------------------------------------------------------------------

// Post notification that the model is done changing. This is different from
// closing from the standpoint that the model could still receive events.

WF_API void WFModels_Model_serverPostDone( WFModels_Model * model );

//------------------------------------------------------------------------------

// Post an event to a model. ( client -> server )

WF_API void WFModels_Model_clientPostEvent(
			WFModels_Model* model,
			unsigned int options,
			const char* selector,
			WFModels_ClientData * data );

#define WFModels_EventOptions_Interlock 1
		/* NOT YET */

//------------------------------------------------------------------------------

// We can temporarily suspend updates from a model. This is purely a client
// side action and will not trigger any response on the server side. It is useful
// if we simply don't want to listen to updates for a while.

WF_API void WFModels_Model_clientSetSuspendUpdates(
						WFModels_Model* model,
						bool updatesSuspended );


//==============================================================================

/****** NOT YET ******/

// Hibernate a model.

WF_API void WFModels_Model_clientHibernate( WFModels_Model* model ); /* NOT YET */

// Revive a model.

WF_API void WFModels_Model_clientRevive( WFModels_Model * model );
			
//==============================================================================

#endif // WFModels_h
