/*******************************************************************************

WFSortOrderStrings.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2013 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

#ifndef WFSortOrderStrings_h
#define WFSortOrderStrings_h

#include <stddef.h>
#include <stdint.h>

#include "WFApiDefs.h"


//------------------------------------------------------------------------------

typedef struct WFSortOrderStrings_Encoding WFSortOrderStrings_Encoding;

//------------------------------------------------------------------------------
/**
	\brief Create a sort order encoding used by the insertion routines.
	
	\param nCodePoints how many code points are used?
	\param decode translation from byte values to ordinal numeric values
	\param encode translation from ordinal numeric values to byte values

	This function validates the raw encoding. We use \c decode to translate
	characters in the input string to numeric values between 0 and \c nCodePoints
	inclusive. We do not encode anything as zero, so we know that \c nCodePoints
	has to be less than \c 0xFF and hence we use decoded values greater than
	\c nCodePoints to signal an error.

	This function verifies that \c decode and \c encode are proper inverses
	of each other.
	
	\return a sort order encoding object used by the insertion routines
		in this namespace
*/

WF_API
WFSortOrderStrings_Encoding* WFSortOrderStrings_makeEncoding( size_t nCodePoints,
															  uint8_t* decode,
															  uint8_t* encode )
		#ifdef __clang__
			__attribute__( ( nonnull( 2, 3 ) ) )
		#endif
	;

//------------------------------------------------------------------------------
/**
	\brief Bias to use when generating a value between two values.
	
	By default middle is a good choice, but if you know you are at the end of a
	range, then low is probably better and if you know you are at the beginning
	of the range, then high is probably better.
*/

typedef enum WFSortOrderStrings_InsertBias {
	ib_low,
	ib_middle,
	ib_high
} WFSortOrderStrings_InsertBias;

//------------------------------------------------------------------------------
/**
	\brief Result / error code for the insertion functions.
*/

typedef enum WFSortOrderStrings_InsertResult {
	ir_ok,
	ir_leftRightMatch,
	ir_leftRightOutOfOrder,
	ir_bufferExhausted,
	ir_invalidCharacter,
	ir_invalidArguments
} WFSortOrderStrings_InsertResult;

//------------------------------------------------------------------------------
/**
	\brief Construct a string lying between the beginning of the encoding range
		and \c right.
*/

WF_API
WFSortOrderStrings_InsertResult
	WFSortOrderStrings_insertBefore( const uint8_t* right,
								     uint8_t* buffer,
								     size_t bufferLimit,
								     const WFSortOrderStrings_Encoding* encoding,
								     WFSortOrderStrings_InsertBias bias )
		#ifdef __clang__
			__attribute__( ( nonnull( 1, 2, 4 ) ) )
		#endif
	;

//------------------------------------------------------------------------------
/**
	\brief Construct a string lying between left and the end of the encoding range.
*/

WF_API
WFSortOrderStrings_InsertResult
	WFSortOrderStrings_insertAfter( const uint8_t* left,
								    uint8_t* buffer,
								    size_t bufferLimit,
								    const WFSortOrderStrings_Encoding* encoding,
								    WFSortOrderStrings_InsertBias bias )
		#ifdef __clang__
			__attribute__( ( nonnull( 1, 2, 4 ) ) )
		#endif
	;

//------------------------------------------------------------------------------
/**
	\brief Construct a string lying between the two strings.
*/

WF_API
WFSortOrderStrings_InsertResult
	WFSortOrderStrings_insertBetween( const uint8_t* left,
									  const uint8_t* right,
									  uint8_t* buffer,
									  size_t bufferLimit,
									  const WFSortOrderStrings_Encoding* encoding,
									  WFSortOrderStrings_InsertBias bias )
		#ifdef __clang__
			__attribute__( ( nonnull( 1, 2, 3, 5 ) ) )
		#endif
	;

//------------------------------------------------------------------------------

#endif
