/*******************************************************************************

blonde.h

********************************************************************************

ADOBE CONFIDENTIAL
------------------
Copyright 2010 Adobe Systems Incorporated
All Rights Reserved.

NOTICE: All information contained herein is, and remain the property of
Adobe Systems Incorporated and its suppliers, if any. The intellectual and
technical concepts contained herein are proprietary to Adobe Systems
Incorporated and its suppliers and may be covered by U.S. and Foreign
Patents, patents in process, and are protected by trade secret or copyright
law.  Dissemination of this information or reproduction of this material is
strictly forbidden unless prior written permission is obtained from
Adobe Systems Incorporated.

*******************************************************************************/

#ifndef _blonde_h
#define _blonde_h

#include "WFApiDefs.h"
#include <AgKernel/lua.h>
#include <AgKernel/AgTransitQueue.h>


//==============================================================================

#pragma mark -
#pragma mark Overview

/*!
	\brief 	BLONDE (Binary Lua Object Notation for Data Exchange) defines a terse binary
	data protocol for encoding and decoding Lua values. It is intended for use as a transit
	format both within a process and across processes.


	The BLONDE library provides functions for encoding and decoding, as shown here
	in Lua pseudo-code:

		local blonde = require 'core.blonde'

		local valueEnc = blonde.encode( value, ... )
			-- Converts a single Lua value into a BLONDE byte stream (a string).
			-- The result is encoded as a BLONDE_TUPLE unless exactly
			-- one argument is passed to this function.

		local value = blonde.decode( valueEnc )
			-- Converts a BLONDE byte stream (a string) into a single Lua value.

	WARNING: When using BLONDE byte streams in C language, you must be careful
	to use the length provided by lua_objlen() and not the length computed by
	strlen(). The binary encoding does not use the zero byte as a terminator.

	A BLONDE bytestream represents any number of Lua values, of the standard Lua
	types nil, boolean, number, string, or table. You can create custom encoders to
	support additional types.

	Tables can be nested up to a reasonable depth (to guard against infinite recursion).
	A table can contain any of the supported data types as keys and values.
        The protocol does not support referencing the same
        value multiple times within a table. When encoding, the library does not check
	for this condition; it re-encodes a value as many times as it appears.

*/


//==============================================================================

#pragma mark -
#pragma mark KSBlondeValue (Native decode value type)

/*
	When using the completely native BLONDE API, a generic wrapper value must be
	returned from the decode function, since we do not have the luxury of dynamic
	typing. A KSBlondeValue is a C struct with two fields: an int type, as
	enumerated below; and a union-ed typedData field, which stores either an int,
	double, or string pointer.

	Suggested use in pseudo code:

	-----

	KSBlondeValue* decodedValue = blonde_decode_c( data );

	switch( decodedValue->type ) {

		case KS_BLONDE_TYPE_INT:
			int myInt = decodedValue->typedData.intValue;
			// do stuff

		case KS_BLONDE_TYPE_DOUBLE:
			double myDouble = decodedValue->typedData.doubleValue;
			// do stuff

		case KS_BLONDE_TYPE_STRING:
			 char* myStr = decodedValue->typedData.stringValue;
			 // do stuff

		case KS_BLONDE_TYPE_BOOLEAN:
			int myBool = decodedValue->typedData.intValue;
			// do stuff

		case KS_BLONDE_TYPE_NULL:
			// No data to retrieve
			// Uh... don't do stuff?

		case KS_BLONDE_TYPE_DECODE_ERROR:
			// There was an error in decoding
			char* msg = decodedValue->typedData.stringValue;
			logError( msg );
	}

	free_KSBlondeValue( decodedValue )

	-----

 */

#if AgEnv_CPP
extern "C" {
#endif

#define KS_BLONDE_TYPE_DECODE_ERROR -1 // used to indicate that an error occurred in decoding
									   // a message can be found in the stringValue field.
#define KS_BLONDE_TYPE_INT 1
#define KS_BLONDE_TYPE_DOUBLE 2
#define KS_BLONDE_TYPE_STRING 3
#define KS_BLONDE_TYPE_BOOLEAN 4 // uses intValue field
#define KS_BLONDE_TYPE_NULL 5 // does not use a data field

typedef struct KSBlondeValue {

	int type;

	union {
		int intValue;
		double doubleValue;
		char* stringValue;
	} typedData;

} KSBlondeValue;

//------------------------------------------------------------------------------

// A handy macro for safely free-ing a KSBlondeValue that has been returned by
// the native decode APIs. This checks if the stringValue field is used and
// frees that as well, if needed.  It uses blonde_free, our wrapper around free()
// which allows for save memory management from different linkage units.

#define free_KSBlondeValue( val ) \
	if( val->type == KS_BLONDE_TYPE_STRING || val->type == KS_BLONDE_TYPE_DECODE_ERROR ) \
		blonde_free( val->typedData.stringValue ); \
	blonde_free( val );\

//==============================================================================

#pragma mark -
#pragma mark Byte stream format

/*
	BLONDE's binary data format uses one byte to encode the data format of the value
	followed by as many bytes are as necessary to complete the representation of
	the value.

	Tables are thus encoded as a stream that looks like this:
	"begin table, key, value, key, value, ..., end table". If any key or value
	is itself a table, then this encoding is applied recursively.

	The data type byte codes (and, if applicable, the data that follows them)
	are as listed below.

	NOTE: Since many hex-dump tools also print ASCII-range characters next to the
	output, we use lower-case letters to specify the data types. Note that the
	letters 'a' through 'd' are off-limits because the overlap the 1-byte int
	range.
*/

//------------------------------------------------------------------------------

#define BLONDE_1_BYTE_INT_MIN -100
#define BLONDE_1_BYTE_INT_MAX 100
	// Any number value that is an integer in the range -100 .. +100 inclusive
	// is encoded directly as a one-byte signed int.

	// Additional data after type code: None.

//------------------------------------------------------------------------------

#define BLONDE_2_BYTE_INT 0x69 // 'i'
	// Any number value that can be represented as a 16-bit signed integer
	// (i.e. -32768 .. +32767), but is outside the 1-byte range described above
	// uses this data type.

	// Additional data after type code (2 bytes): Little-endian 16-bit signed int.

//------------------------------------------------------------------------------

#define	BLONDE_4_BYTE_INT 0x6C // 'l'
	// Any number value that can be represented as a 32-bit signed integer
	// (i.e. -2147483648 .. +2147483647), but is outside the 2-byte range described
	// uses this data type.

	// Additional data after type code (4 bytes): Little-endian 32-bit signed int.

//------------------------------------------------------------------------------

#define BLONDE_NUMBER 0x6E // 'n'
	// Any number value that can not be represented by the integer types above
	// (i.e. non-integers and integers outside the 32-bit signed space)
	// uses this data type.

	// Additional data after type code (8 bytes): Little-endian double value.

//------------------------------------------------------------------------------

#define BLONDE_FALSE 0x66 // 'f'
	// Boolean false.
	// Additional data after type code: None.

//------------------------------------------------------------------------------

#define BLONDE_TRUE 0x74 // 't'
	// Boolean true.
	// Additional data after type code: None.

//------------------------------------------------------------------------------

#define BLONDE_SHORT_STRING -128 // a.k.a. 0x80
#define BLONDE_SHORT_STRING_MAX_LENGTH ( -1 + BLONDE_1_BYTE_INT_MIN - BLONDE_SHORT_STRING )
	// Strings of 27 bytes or fewer are encoded using this data type. The length
	// of the string is added to BLONDE_SHORT_STRING and used as the data type.

	// Additional data after type code:
		// String data (# bytes = type code byte - 128), not NULL terminated.

	// Example: a five-byte string value would be encoded as the type code 133 (=128 + 5)
	// followed by the five bytes of the string.

//------------------------------------------------------------------------------

#define BLONDE_LONG_STRING 0x73 // 's'
	// Strings of 28 bytes or longer are encoded using this data type.
	// The length of the string is encoded using the numeric encoding above,
	// and that that many bytes of string data follow.

	// Additional data after type code:
		// String length (encoded using numeric types above)
		// String data, not NULL-terminated

	// WARNING: While this encoding theoretically allows you to describe
	// strings of gigabytes or even longer length, you probably shouldn't do
	// that. Remember that this all has to fit (twice) in memory somehow.

//------------------------------------------------------------------------------

#define BLONDE_TABLE 0x6B // 'k'
	// Tables are encoded using this data type.

	// Additional data after type code:
		// Key (encoded using any data type represented here, except table)
		// Value (encoded using any data type represented here)
		// (Repeat until all key-value pairs have been encoded)
		// BLONDE_TABLE_END

	// In order to maintain predictable/comparable output data, the encoder will
	// write the key-value pairs in sorted order, using the following sort sequence
	// (and omitting any keys from this list that aren't present in the table):

		// boolean false
		// boolean true
		// numeric values (lowest to highest)
		// string values (lowest to highest as specified by Lua string comparison)

	// IMPORTANT: Tables may be used as values within another table, but not as
	// keys. Such values are simply encoded recursively using this same mechanism.
	// The encoder is not smart about tables that share a reference to a subtable.
	// If it encounters such, it will re-encode them. It does, however, enforce
	// an arbitrary depth limit (currently hard-coded to 20) as a guard against
	// infinite recursion.

		// GRUNGE [scouten 2011-02-01]: Can we remove the tables as keys restriction
		// in the case where table keys are not sorted?

	// IMPORTANT: Metatables are NOT encoded. Also, values that are only visible
	// in Lua via the use of an __index metamethod will probably not be encoded.

	// TBD: Should the encoder raise an error if it's asked to encode a table
	// that has a metatable?

//------------------------------------------------------------------------------

#define BLONDE_TABLE_END 0x78 // 'x'
	// Marks the end of a table.

//------------------------------------------------------------------------------

#define BLONDE_NIL 0x7A // 'z'
	// Only used if the top-level value passed to blonde_encode is itself nil.
	// Otherwise never appears.

//------------------------------------------------------------------------------

#define BLONDE_TUPLE 0x70 // 'p'
	// Only used if blonde_encode is called with BLONDE_ENCODE_ALL or
	// BLONDE_ENCODE_FROM as the index. Represents the series of values on
	// the stack at that time (and thus the series of values to be unpacked
	// at decode time).

	// Additional data after type code:
		// Number of values (encoded using integer values above)
		// Values (encoded using any data type represented here)

//------------------------------------------------------------------------------

#define BLONDE_USERDATA 0x75 // 'u'
	// Additional data after type code:
		// Userdata length (encoded using numeric types above)
		// Userdata data, not NULL-terminated

typedef void( *blonde_writeDataFunc )( void* blondeContext, const void* data, size_t nBytes );
	// This function is passed to the user-supplied blonde_encoderFunc below.
	// It must be called once and only once per encoder invocation during each
	// of the passes (measure and encode) because it needs an accurate count of
	// data size as a prefix to the data). When called during the encode pass,
	// it copies the data from your buffer into the BLONDE byte stream before
	// returning. Thus it is encouraged, when possible, to use a stack-based
	// buffer for encoding.

	// Your encoder function must pass the void* pointer "blondeContext" back
	// to this function.

typedef void( *blonde_encoderFunc )( lua_State* L, int index, void* encoderFuncContext,
					int measurePass, blonde_writeDataFunc writeDataFunc,
					void* blondeContext );
	// If this function is provided, it will be called during blonde_encode for
	// any Lua values that can not be encoded directly by BLONDE (i.e. userdata,
	// function, or thread). If your function can encode the value, it must call
	// the blonde_writeDataFunc callback once and only once. If it can not
	// encode the value, it should simply not call the writeDataFunc and this will
	// trigger an "invalid value" error.

	// Values encoded in this fashion are expected to be used for intra-process
	// communication only. You probably should not provide an encoder function
	// when recording BLONDE streams for persistence or inter-process communication.

	// Your function must be careful not to change any values on the Lua stack.

	// Implementation note: BLONDE encodes values by scanning twice: once to
	// determine the size of the buffer that will be needed (i.e. the "measure
	// pass") and a second time to populate that buffer. When measurePass is
	// true (non-zero), the writeDataFunc will not actually copy data, but will
	// simply record the number of bytes specified. If your encoder can take
	// shortcuts to measuring the data payload, it is encouraged to do so.

typedef void( *blonde_decoderFunc )( lua_State* L, void* decoderFuncContext,
					const void* data, size_t nBytes );
	// If this function is provided, it will be called during blonde_decode for
	// any Lua values that were encoded using the (presumably) matching encoder
	// function. The function should convert the serialized data and bytes
	// into a corresponding Lua value and push exactly one value onto the stack.

	// GRUNGE [scouten 2011-01-31]: These comments need better wording.

typedef void* ( *blonde_outputFunc )( void* outputFuncContext,
                    void** data, size_t nBytes );
    // The data to be output at the end of the encoding process. The output
    // function is free to take control of the data (which was allocated) by
    // malloc by setting *data to NULL. If this does not happen, then the data
    // will be freed by the blonde encoder logic upon return from the output
    // function. The result of the output function will also be the result
    // of the encode function.

//------------------------------------------------------------------------------

#define BLONDE_API_VERSION 0x79 // 'y'


//==============================================================================

#pragma mark -
#pragma mark Encoding/decoding API (C language)

//------------------------------------------------------------------------------

/*!
	Converts a single Lua value into a BLONDE byte stream
	and leaves the encoded value as a string at the top of the stack.

 	Raises a Lua error if the value is (or contains)
    	an invalid value type (such as a function, userdata, or thread).

    	Because the string value can contain NULL, you must use lua_objlen
    	(rather than strlen) to retrieve its size.

	\param L  The Lua state
	\param index The stack index of the value to convert
*/

WF_API void blonde_encode( lua_State* L, int index );

#define BLONDE_ENCODE_ALL (-20001)
	// You may pass BLONDE_ENCODE_ALL as the value for index to any of the
	// blonde_encode* functions, in which case all values on the Lua stack are
	// encoded as a BLONDE_TUPLE.

#define BLONDE_ENCODE_FROM(index) (-20000 - (index))
	// You may pass BLONDE_ENCODE_FROM( index ) as the value for index to any
	// of the blonde_encode* functions, in which case all values starting with
	// index are encoded as a BLONDE_TUPLE.

//------------------------------------------------------------------------------

/*!
	Converts a single Lua value into a BLONDE byte stream
	and leaves the encoded value as a string at the top of the stack.
	For table values, sorts the keys to ensure stable, consistent output
	(and therefore takes longer then blonde_encode()).

 	Raises a Lua error if the value is (or contains)
    	an invalid value type (such as a function, userdata, or thread).

    	Because the string value can contain NULL, you must use lua_objlen
    	(rather than strlen) to retrieve its size.

	\param L  The Lua state
	\param index The stack index of the value to convert
*/

WF_API void blonde_encode_stable_output( lua_State* L, int index );

//------------------------------------------------------------------------------

WF_API void* blonde_encodeWithOptions( lua_State* L, int index, int sortTableKeys,
					blonde_encoderFunc encoderFunc, void* encoderFuncContext,
                    blonde_outputFunc outputFunc, void* outputFuncContext );
	// As above, but accepts an encoding function that will be passed any
	// values that can't be encoded directly through BLONDE's built-in handlers
	// (i.e. userdata, functions, and threads). This function gets a chance
	// to encode such values.

//------------------------------------------------------------------------------

WF_API int blonde_decodeString( lua_State* L, int index );
	// Takes the value at the specified stack index and decodes it using the
	// BLONDE protocol described above. Leaves the decoded value at the top of
	// the stack. Raises a Lua error if the value is not a string, or if it can
	// not be parsed properly.

	// Returns the number of values placed on the Lua stack. (Typically one
	// but may be different if a BLONDE_TUPLE was encoded.)

//------------------------------------------------------------------------------

WF_API int blonde_decodeStringWithOptions( lua_State* L, int index,
					blonde_decoderFunc decoderFunc, void* decoderFuncContext );
	// As above, but accepts an decoding function that will be passed any
	// values encoded via an earlier blonde_encoderFunc (i.e. userdata, functions,
	// and threads). This function gets a chance to decode such values.
	// The decoderFunc may be NULL.

	// Returns the number of values placed on the Lua stack. (Typically one
	// but may be different if a BLONDE_TUPLE was encoded.)


//------------------------------------------------------------------------------

WF_API int blonde_decodeDataWithOptions( lua_State* L,
					const void* dataPtr, size_t dataSize,
					blonde_decoderFunc decoderFunc, void* decoderFuncContext );
	// As above, but rather than taking the data as an object on the Lua
	// stack, this expects a pointer to the data and its legth.

	// Returns the number of values placed on the Lua stack. (Typically one
	// but may be different if a BLONDE_TUPLE was encoded.)


//==============================================================================

#pragma mark -
#pragma mark Encoding/decoding API (C language native)

//------------------------------------------------------------------------------

// WARNING: The string value may contain NULL, and will not necessarily be
// NULL-terminated, so strlen is useless.

// REVIEW [epoley 2011-01-25]: Do we wish to return the string length as well in
// case the client wishes to do something like copy it? Or should that remain
// unsupported? Or, we could encode the total string length in the bytestream
// and provide a function for determining a given BLONDE stream's length.

// REVIEW [scouten 2011-03-02]: Do we actually use these? Perhaps these could
// also move to incubator?

// REVIEW [erwright 2011-06-01]: We're starting to use them now.

// IMPORTANT: You are responsible for freeing this string using blonde_free()
// when you are done with it.
WF_API void blonde_free( void * value );

WF_API const char* blonde_encode_string( const char* value, size_t* encodedLength );

WF_API const char* blonde_encode_int( int value );

WF_API const char* blonde_encode_boolean( int value );

WF_API const char* blonde_encode_double( double value );

WF_API const char* blonde_encode_null( void );

WF_API const char* blonde_encode_transit_cargo( AgTransitCargoPtr cargo, size_t* encodedLength );

//------------------------------------------------------------------------------

/*!
	Converts a BLONDE byte stream (a string) into a single Lua value.

	Returns the result of the decode operation as a KSBlondeValue structure,
	which has two fields: an int representing the type of data, and
	a union of typed data. The type value helps you determine which field
 	to access.

	 The function allocates this structure, but you are responsible for freeing
	 both the structure and its string data when it is no longer needed,
	 using the macro free_KSBlondeValue() .

 	\return The result of decoding, as a KSBlondeValue structure, or an error.

*/

WF_API KSBlondeValue* blonde_decode_c( const char* data );

//==============================================================================

#pragma mark -
#pragma mark Encoding/decoding API (Lua language)

//------------------------------------------------------------------------------

WF_API int luaopen_blondeImp( lua_State* L );
	// This conforms to the standard Lua package loader protocol. Pushes a table
	// onto the Lua stack that contains functions named "encode" and "decode"
	// that do the same as the C functions above. The usage pattern is as in
	// the sample code at the top of this file.

//------------------------------------------------------------------------------

#if AgEnv_CPP
} // extern "C"
#endif

#endif // _blonde_h
