//==============================================================================
//
//  WFOzAssetInfo.h
//
//==============================================================================
//
//	ADOBE CONFIDENTIAL
//	------------------
//	Copyright 2013 Adobe Systems Incorporated
//	All Rights Reserved.
//
//	NOTICE: All information contained herein is, and remains the property of
//	Adobe Systems Incorporated and its suppliers, if any. The intellectual and
//	technical concepts contained herein are proprietary to Adobe Systems
//	Incorporated and its suppliers and are protected by all applicable
//	intellectual property laws, including trade secret and copyright laws.
//	Dissemination of this information or reproduction of this material is
//	strictly forbidden unless prior written permission is obtained from
//	Adobe Systems Incorporated.
//
//==============================================================================

/*!
	\file WFOzAssetInfo.h

	\brief \c WFOzAssetInfo provides an efficient encoding of the core info regarding
	an oz asset.  Each info struct occupies approximately 40 bytes of storage.
	
	This could be further optimized in the future by finding a way to exclude
	GUIDs (in favor of local integer ids, for example).
*/

#ifndef wf_oz_client_WFOzAssetInfo_h
#define wf_oz_client_WFOzAssetInfo_h

#include <AgKernel/AgTypes.h>
#include <AgKernel/lua.h>
#include <WFCore/WFApiDefs.h>

//==============================================================================

#pragma mark -
#pragma mark typedefs

//------------------------------------------------------------------------------

struct WFOzAssetInfo_struct {
	unsigned char _id[ 16 ];		 	/**< Compact representation of 32-char GUID asset id */
	unsigned char _thumbnailMD5[ 16 ];	/**< Compact representation of 32-char char thumb md5 */
	float _aspectRatio;					/**< Aspect ratio of cropped asset */
	uint16_t _duration;					/**< Duration (in seconds) */
	unsigned int _hasThumbnailMd5 : 1;	/**< Did we get thumbnail md5 */
	unsigned int _hasDuration : 1;		/**< Did we get duration */
	unsigned int _isVideo : 1;			/**< Is the asset a video */
};

typedef struct WFOzAssetInfo_struct * WFOzAssetInfoPtr;
	/**<
		\brief The asset info contains compact forms of the id thumbnail MD5,
		the capture date, aspect ratio, duration and several yes/no values.
	*/

typedef struct WFOzAssetInfoArray_struct * WFOzAssetInfoArrayPtr;
	/**<
		\brief The asset info array is a way to represent multiple asset infos
		as a single allocation.  Additionally, it is compatible with the
		WFLuaRCObject APIs.
	*/

//------------------------------------------------------------------------------

void WFOzAssetInfo_decodeGuid( char *uuid, const unsigned char *data );
	/**<
		\brief Decodes a compact GUID back to ASCII lowercase hex string.

		You must pass preallocated space (33 bytes) for the result in uuid.
		data must be 16 bytes in size.
		
		the last byte of uuid (33) will be set to '\0' to make it a valid null
		terminated "C" string.

		\param uuid a non-NULL pointer for the output characters.
		\param data a non-NULL pointer for the encoded bytes.
	*/

//--------------------------------------------------------------------------

int WFOzAssetInfo_compactGuid( unsigned char *data, const char *guid );
	/**<
		\brief Encodes a hexidecimal GUID (upper/lower/mixed case) in half
		as many bytes by converting it to binary form.

		You must pass preallocated space (16 bytes) for the result in data.
		guid must point to a "C" string that contains at least 32 hex chars.
		(Additional chars are ignored). It may optinally contain '-' separators
		which will be ignored.  (When decoding a compacted GUID the hyphens
		are NOT retained.)

		\param data a non-NULL pointer for the output bytes.
		\param guid a non-NULL pointer for the input characters.
	*/

//--------------------------------------------------------------------------

WF_API WFOzAssetInfoArrayPtr WFOzAssetInfoArray_optFromLua( lua_State* L, int index );
	/**<
		\brief Retreives an asset info array from lua.  It uses the API
		wflrco_toTempRCObject internally, so if you wish to retain the result
		for a while, be sure to retain it with WFOzAssetInfoArray_retain
		(below)

		\param L the lua state
		\param index the place on the lua state's stack to find the array.
	*/

//--------------------------------------------------------------------------

WF_API int WFOzAssetInfoArray_count( WFOzAssetInfoArrayPtr array );
	/**<
		\brief Get how many asset info objects are held by the array.
		
		\return the count of objects (>= 0).

		\param array the wf asset info array.
	*/

//--------------------------------------------------------------------------

WF_API WFOzAssetInfoArrayPtr WFOzAssetInfoArray_retain( WFOzAssetInfoArrayPtr array );
	/**<
		\brief Add a reference count to the array.
		
		\return the array.

		\param array the wf asset info array.
	*/

//--------------------------------------------------------------------------

WF_API void WFOzAssetInfoArray_release( WFOzAssetInfoArrayPtr array );
	/**<
		\brief Remove a reference count from the array.
		
		\return the array.

		\param array the wf asset info array.
	*/

//--------------------------------------------------------------------------

WF_API WFOzAssetInfoPtr WFOzAssetInfoArray_objectAtIndex( WFOzAssetInfoArrayPtr array, int index );
	/**<
		\brief Get an asset from the array.
		
		\return the asset info.  If index is greater than or equal to the count for the array
		this returns NULL.
		The resulting pointer will be valid as long as the input array is valid (has a positive
		retain count via WFOzAssetInfoArray_retain).

		\param array the wf asset info array.
		\param index the index of the info object to retreive.
	*/

//--------------------------------------------------------------------------

WF_API unsigned char * WFOzAssetInfoArray_encodedHash( WFOzAssetInfoArrayPtr array );
	/**<
		\brief Get an encoded hash for the list.  The list contains a hash which is a
		simple digest of all the asset ids it contains
		
		\return the encoded hash.  use WFOzAssetInfo_decodeGuid to extract the hash
		as a 32-character ASCII hex string.

		\param array the wf asset info array.
	*/

//--------------------------------------------------------------------------

WF_API int WFOzAssetInfoArray_approxDateForAssetAtIndex( WFOzAssetInfoArrayPtr array, int index, char *outString );
	/**<
		\brief copies into char array an approximate date for the asset at index.
		
		\param array the wf asset info array.
		\param index which element to fetch.
		\param outString, the approximate date as a string copied into the preallocated space at outString (of
		at least 21 bytes).
		
		\return true if the date was copied.  Dates are not always available.
	*/

//--------------------------------------------------------------------------

#endif
